/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.Block;
import com.google.bitcoin.core.BlockChainListener;
import com.google.bitcoin.core.FilteredBlock;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.ProtocolException;
import com.google.bitcoin.core.PrunedException;
import com.google.bitcoin.core.ScriptException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.StoredBlock;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.TransactionOutputChanges;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VerificationException;
import com.google.bitcoin.core.Wallet;
import com.google.bitcoin.store.BlockStore;
import com.google.bitcoin.store.BlockStoreException;
import com.google.bitcoin.utils.Locks;
import com.google.common.base.Preconditions;
import java.math.BigInteger;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractBlockChain {
    private static final Logger log = LoggerFactory.getLogger(AbstractBlockChain.class);
    protected ReentrantLock lock = Locks.lock("blockchain");
    private final BlockStore blockStore;
    protected StoredBlock chainHead;
    private final Object chainHeadLock = new Object();
    protected final NetworkParameters params;
    private final CopyOnWriteArrayList<BlockChainListener> listeners;
    private final LinkedHashMap<Sha256Hash, OrphanBlock> orphanBlocks = new LinkedHashMap();
    private long statsLastTime = System.currentTimeMillis();
    private long statsBlocksAdded;
    private static Date testnetDiffDate = new Date(1329264000000L);

    public AbstractBlockChain(NetworkParameters params, List<BlockChainListener> listeners, BlockStore blockStore) throws BlockStoreException {
        this.blockStore = blockStore;
        this.chainHead = blockStore.getChainHead();
        log.info("chain head is at height {}:\n{}", (Object)this.chainHead.getHeight(), (Object)this.chainHead.getHeader());
        this.params = params;
        this.listeners = new CopyOnWriteArrayList<BlockChainListener>(listeners);
    }

    public void addWallet(Wallet wallet) {
        this.listeners.add(wallet);
    }

    public void addListener(BlockChainListener listener) {
        this.listeners.add(listener);
    }

    public void removeListener(BlockChainListener listener) {
        this.listeners.remove(listener);
    }

    public BlockStore getBlockStore() {
        return this.blockStore;
    }

    protected abstract StoredBlock addToBlockStore(StoredBlock var1, Block var2) throws BlockStoreException, VerificationException;

    protected abstract StoredBlock addToBlockStore(StoredBlock var1, Block var2, TransactionOutputChanges var3) throws BlockStoreException, VerificationException;

    protected abstract void doSetChainHead(StoredBlock var1) throws BlockStoreException;

    protected abstract void notSettingChainHead() throws BlockStoreException;

    protected abstract StoredBlock getStoredBlockInCurrentScope(Sha256Hash var1) throws BlockStoreException;

    public boolean add(Block block) throws VerificationException, PrunedException {
        try {
            return this.add(block, null, null, true);
        }
        catch (BlockStoreException e) {
            throw new RuntimeException(e);
        }
        catch (VerificationException e) {
            try {
                this.notSettingChainHead();
            }
            catch (BlockStoreException e1) {
                throw new RuntimeException(e1);
            }
            throw new VerificationException("Could not verify block " + block.getHashAsString() + "\n" + block.toString(), e);
        }
    }

    public boolean add(FilteredBlock block) throws VerificationException, PrunedException {
        try {
            HashSet<Sha256Hash> filteredTxnHashSet = new HashSet<Sha256Hash>(block.getTransactionHashes());
            List<Transaction> filteredTxn = block.getAssociatedTransactions();
            for (Transaction tx : filteredTxn) {
                Preconditions.checkState((boolean)filteredTxnHashSet.remove(tx.getHash()));
            }
            return this.add(block.getBlockHeader(), filteredTxnHashSet, filteredTxn, true);
        }
        catch (BlockStoreException e) {
            throw new RuntimeException(e);
        }
        catch (VerificationException e) {
            try {
                this.notSettingChainHead();
            }
            catch (BlockStoreException e1) {
                throw new RuntimeException(e1);
            }
            throw new VerificationException("Could not verify block " + block.getHash().toString() + "\n" + block.toString(), e);
        }
    }

    protected abstract boolean shouldVerifyTransactions();

    protected abstract TransactionOutputChanges connectTransactions(int var1, Block var2) throws VerificationException, BlockStoreException;

    protected abstract TransactionOutputChanges connectTransactions(StoredBlock var1) throws VerificationException, BlockStoreException, PrunedException;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean add(Block block, Set<Sha256Hash> filteredTxHashList, List<Transaction> filteredTxn, boolean tryConnecting) throws BlockStoreException, VerificationException, PrunedException {
        this.lock.lock();
        try {
            if (System.currentTimeMillis() - this.statsLastTime > 1000L) {
                if (this.statsBlocksAdded > 1L) {
                    log.info("{} blocks per second", (Object)this.statsBlocksAdded);
                }
                this.statsLastTime = System.currentTimeMillis();
                this.statsBlocksAdded = 0L;
            }
            if (block.equals(this.getChainHead().getHeader())) {
                boolean bl = true;
                return bl;
            }
            if (tryConnecting && this.orphanBlocks.containsKey(block.getHash())) {
                boolean bl = false;
                return bl;
            }
            if (this.shouldVerifyTransactions() && block.transactions == null) {
                throw new VerificationException("Got a block header while running in full-block mode");
            }
            boolean contentsImportant = this.shouldVerifyTransactions();
            if (block.transactions != null) {
                contentsImportant = contentsImportant || this.containsRelevantTransactions(block);
            }
            try {
                block.verifyHeader();
                if (contentsImportant) {
                    block.verifyTransactions();
                }
            }
            catch (VerificationException e) {
                log.error("Failed to verify block: ", (Throwable)e);
                log.error(block.getHashAsString());
                throw e;
            }
            StoredBlock storedPrev = this.getStoredBlockInCurrentScope(block.getPrevBlockHash());
            if (storedPrev == null) {
                Preconditions.checkState((boolean)tryConnecting, (Object)"bug in tryConnectingOrphans");
                log.warn("Block does not connect: {} prev {}", (Object)block.getHashAsString(), (Object)block.getPrevBlockHash());
                this.orphanBlocks.put(block.getHash(), new OrphanBlock(block, filteredTxHashList, filteredTxn));
                boolean bl = false;
                return bl;
            }
            this.checkDifficultyTransitions(storedPrev, block);
            this.connectBlock(block, storedPrev, this.shouldVerifyTransactions(), filteredTxHashList, filteredTxn);
            if (tryConnecting) {
                this.tryConnectingOrphans();
            }
            ++this.statsBlocksAdded;
            boolean bl = true;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void connectBlock(Block block, StoredBlock storedPrev, boolean expensiveChecks, Set<Sha256Hash> filteredTxHashList, List<Transaction> filteredTxn) throws BlockStoreException, VerificationException, PrunedException {
        StoredBlock head;
        Preconditions.checkState((boolean)this.lock.isLocked());
        if (!this.params.passesCheckpoint(storedPrev.getHeight() + 1, block.getHash())) {
            throw new VerificationException("Block failed checkpoint lockin at " + (storedPrev.getHeight() + 1));
        }
        if (this.shouldVerifyTransactions()) {
            for (Transaction tx : block.transactions) {
                if (tx.isFinal(storedPrev.getHeight() + 1, block.getTimeSeconds())) continue;
                throw new VerificationException("Block contains non-final transaction");
            }
        }
        if (storedPrev.equals(head = this.getChainHead())) {
            if (expensiveChecks && block.getTimeSeconds() <= AbstractBlockChain.getMedianTimestampOfRecentBlocks(head, this.blockStore)) {
                throw new VerificationException("Block's timestamp is too early");
            }
            TransactionOutputChanges txOutChanges = null;
            if (this.shouldVerifyTransactions()) {
                txOutChanges = this.connectTransactions(storedPrev.getHeight() + 1, block);
            }
            StoredBlock newStoredBlock = this.addToBlockStore(storedPrev, block.transactions == null ? block : block.cloneAsHeader(), txOutChanges);
            this.setChainHead(newStoredBlock);
            log.debug("Chain is now {} blocks high", (Object)newStoredBlock.getHeight());
            boolean first = true;
            for (BlockChainListener listener : this.listeners) {
                if (block.transactions != null || filteredTxn != null) {
                    AbstractBlockChain.sendTransactionsToListener(newStoredBlock, NewBlockType.BEST_CHAIN, listener, block.transactions != null ? block.transactions : filteredTxn, !first);
                }
                if (filteredTxHashList != null) {
                    for (Sha256Hash hash : filteredTxHashList) {
                        listener.notifyTransactionIsInBlock(hash, newStoredBlock, NewBlockType.BEST_CHAIN);
                    }
                }
                listener.notifyNewBestBlock(newStoredBlock);
                first = false;
            }
        } else {
            StoredBlock newBlock = storedPrev.build(block);
            boolean haveNewBestChain = newBlock.moreWorkThan(head);
            if (haveNewBestChain) {
                log.info("Block is causing a re-organize");
            } else {
                StoredBlock splitPoint = AbstractBlockChain.findSplit(newBlock, head, this.blockStore);
                if (splitPoint != null && splitPoint.equals(newBlock)) {
                    log.warn("Saw duplicated block in main chain at height {}: {}", (Object)newBlock.getHeight(), (Object)newBlock.getHeader().getHash());
                    return;
                }
                if (splitPoint == null) {
                    throw new VerificationException("Block forks the chain but splitPoint is null");
                }
                this.addToBlockStore(storedPrev, block);
                int splitPointHeight = splitPoint.getHeight();
                String splitPointHash = splitPoint.getHeader().getHashAsString();
                log.info("Block forks the chain at height {}/block {}, but it did not cause a reorganize:\n{}", new Object[]{splitPointHeight, splitPointHash, newBlock.getHeader().getHashAsString()});
            }
            if (block.transactions != null || filteredTxn != null) {
                boolean first = true;
                for (BlockChainListener listener : this.listeners) {
                    List<Transaction> txnToNotify = block.transactions != null ? block.transactions : filteredTxn;
                    AbstractBlockChain.sendTransactionsToListener(newBlock, NewBlockType.SIDE_CHAIN, listener, txnToNotify, first);
                    if (filteredTxHashList != null) {
                        for (Sha256Hash hash : filteredTxHashList) {
                            listener.notifyTransactionIsInBlock(hash, newBlock, NewBlockType.SIDE_CHAIN);
                        }
                    }
                    first = false;
                }
            }
            if (haveNewBestChain) {
                this.handleNewBestChain(storedPrev, newBlock, block, expensiveChecks);
            }
        }
    }

    private static long getMedianTimestampOfRecentBlocks(StoredBlock storedBlock, BlockStore store) throws BlockStoreException {
        long[] timestamps = new long[11];
        int unused = 9;
        timestamps[10] = storedBlock.getHeader().getTimeSeconds();
        while (unused >= 0 && (storedBlock = storedBlock.getPrev(store)) != null) {
            timestamps[unused--] = storedBlock.getHeader().getTimeSeconds();
        }
        Arrays.sort(timestamps, unused + 1, 11);
        return timestamps[unused + (11 - unused) / 2];
    }

    protected abstract void disconnectTransactions(StoredBlock var1) throws PrunedException, BlockStoreException;

    private void handleNewBestChain(StoredBlock storedPrev, StoredBlock newChainHead, Block block, boolean expensiveChecks) throws BlockStoreException, VerificationException, PrunedException {
        Preconditions.checkState((boolean)this.lock.isLocked());
        StoredBlock head = this.getChainHead();
        StoredBlock splitPoint = AbstractBlockChain.findSplit(newChainHead, head, this.blockStore);
        log.info("Re-organize after split at height {}", (Object)splitPoint.getHeight());
        log.info("Old chain head: {}", (Object)head.getHeader().getHashAsString());
        log.info("New chain head: {}", (Object)newChainHead.getHeader().getHashAsString());
        log.info("Split at block: {}", (Object)splitPoint.getHeader().getHashAsString());
        LinkedList<StoredBlock> oldBlocks = AbstractBlockChain.getPartialChain(head, splitPoint, this.blockStore);
        LinkedList<StoredBlock> newBlocks = AbstractBlockChain.getPartialChain(newChainHead, splitPoint, this.blockStore);
        StoredBlock storedNewHead = splitPoint;
        if (this.shouldVerifyTransactions()) {
            for (StoredBlock oldBlock : oldBlocks) {
                this.disconnectTransactions(oldBlock);
            }
            Iterator<StoredBlock> it = newBlocks.descendingIterator();
            while (it.hasNext()) {
                StoredBlock cursor = it.next();
                if (expensiveChecks && cursor.getHeader().getTimeSeconds() <= AbstractBlockChain.getMedianTimestampOfRecentBlocks(cursor.getPrev(this.blockStore), this.blockStore)) {
                    throw new VerificationException("Block's timestamp is too early during reorg");
                }
                TransactionOutputChanges txOutChanges = cursor != newChainHead || block == null ? this.connectTransactions(cursor) : this.connectTransactions(newChainHead.getHeight(), block);
                storedNewHead = this.addToBlockStore(storedNewHead, cursor.getHeader(), txOutChanges);
            }
        } else {
            storedNewHead = this.addToBlockStore(storedPrev, newChainHead.getHeader());
        }
        for (int i = 0; i < this.listeners.size(); ++i) {
            BlockChainListener listener = this.listeners.get(i);
            listener.reorganize(splitPoint, oldBlocks, newBlocks);
            if (i == this.listeners.size()) break;
            if (this.listeners.get(i) == listener) continue;
            --i;
        }
        this.setChainHead(storedNewHead);
    }

    private static LinkedList<StoredBlock> getPartialChain(StoredBlock higher, StoredBlock lower, BlockStore store) throws BlockStoreException {
        Preconditions.checkArgument((higher.getHeight() > lower.getHeight() ? 1 : 0) != 0, (Object)"higher and lower are reversed");
        LinkedList<StoredBlock> results = new LinkedList<StoredBlock>();
        StoredBlock cursor = higher;
        do {
            results.add(cursor);
        } while (!(cursor = (StoredBlock)Preconditions.checkNotNull((Object)cursor.getPrev(store), (Object)"Ran off the end of the chain")).equals(lower));
        return results;
    }

    private static StoredBlock findSplit(StoredBlock newChainHead, StoredBlock oldChainHead, BlockStore store) throws BlockStoreException {
        StoredBlock currentChainCursor = oldChainHead;
        StoredBlock newChainCursor = newChainHead;
        while (!currentChainCursor.equals(newChainCursor)) {
            if (currentChainCursor.getHeight() > newChainCursor.getHeight()) {
                currentChainCursor = currentChainCursor.getPrev(store);
                Preconditions.checkNotNull((Object)currentChainCursor, (Object)"Attempt to follow an orphan chain");
                continue;
            }
            newChainCursor = newChainCursor.getPrev(store);
            Preconditions.checkNotNull((Object)newChainCursor, (Object)"Attempt to follow an orphan chain");
        }
        return currentChainCursor;
    }

    public int getBestChainHeight() {
        return this.getChainHead().getHeight();
    }

    private static void sendTransactionsToListener(StoredBlock block, NewBlockType blockType, BlockChainListener listener, List<Transaction> transactions, boolean clone) throws VerificationException {
        for (Transaction tx : transactions) {
            try {
                if (!listener.isTransactionRelevant(tx)) continue;
                if (clone) {
                    tx = new Transaction(tx.params, tx.bitcoinSerialize());
                }
                listener.receiveFromBlock(tx, block, blockType);
            }
            catch (ScriptException e) {
                log.warn("Failed to parse a script: " + e.toString());
            }
            catch (ProtocolException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void setChainHead(StoredBlock chainHead) throws BlockStoreException {
        this.doSetChainHead(chainHead);
        Object object = this.chainHeadLock;
        synchronized (object) {
            this.chainHead = chainHead;
        }
    }

    private void tryConnectingOrphans() throws VerificationException, BlockStoreException, PrunedException {
        int blocksConnectedThisRound;
        Preconditions.checkState((boolean)this.lock.isLocked());
        do {
            blocksConnectedThisRound = 0;
            Iterator<OrphanBlock> iter = this.orphanBlocks.values().iterator();
            while (iter.hasNext()) {
                OrphanBlock orphanBlock = iter.next();
                log.debug("Trying to connect {}", (Object)orphanBlock.block.getHash());
                StoredBlock prev = this.getStoredBlockInCurrentScope(orphanBlock.block.getPrevBlockHash());
                if (prev == null) {
                    log.debug("  but it is not connectable right now");
                    continue;
                }
                this.add(orphanBlock.block, orphanBlock.filteredTxHashes, orphanBlock.filteredTxn, false);
                iter.remove();
                ++blocksConnectedThisRound;
            }
            if (blocksConnectedThisRound <= 0) continue;
            log.info("Connected {} orphan blocks.", (Object)blocksConnectedThisRound);
        } while (blocksConnectedThisRound > 0);
    }

    private void checkDifficultyTransitions(StoredBlock storedPrev, Block nextBlock) throws BlockStoreException, VerificationException {
        Preconditions.checkState((boolean)this.lock.isLocked());
        Block prev = storedPrev.getHeader();
        if ((storedPrev.getHeight() + 1) % this.params.interval != 0) {
            if (this.params.getId().equals("org.bitcoin.test") && nextBlock.getTime().after(testnetDiffDate)) {
                this.checkTestnetDifficulty(storedPrev, prev, nextBlock);
                return;
            }
            if (nextBlock.getDifficultyTarget() != prev.getDifficultyTarget()) {
                throw new VerificationException("Unexpected change in difficulty at height " + storedPrev.getHeight() + ": " + Long.toHexString(nextBlock.getDifficultyTarget()) + " vs " + Long.toHexString(prev.getDifficultyTarget()));
            }
            return;
        }
        long now = System.currentTimeMillis();
        StoredBlock cursor = this.blockStore.get(prev.getHash());
        for (int i = 0; i < this.params.interval - 1; ++i) {
            if (cursor == null) {
                throw new VerificationException("Difficulty transition point but we did not find a way back to the genesis block.");
            }
            cursor = this.blockStore.get(cursor.getHeader().getPrevBlockHash());
        }
        long elapsed = System.currentTimeMillis() - now;
        if (elapsed > 50L) {
            log.info("Difficulty transition traversal took {}msec", (Object)elapsed);
        }
        Block blockIntervalAgo = cursor.getHeader();
        int timespan = (int)(prev.getTimeSeconds() - blockIntervalAgo.getTimeSeconds());
        if (timespan < this.params.targetTimespan / 4) {
            timespan = this.params.targetTimespan / 4;
        }
        if (timespan > this.params.targetTimespan * 4) {
            timespan = this.params.targetTimespan * 4;
        }
        BigInteger newDifficulty = Utils.decodeCompactBits(prev.getDifficultyTarget());
        newDifficulty = newDifficulty.multiply(BigInteger.valueOf(timespan));
        if ((newDifficulty = newDifficulty.divide(BigInteger.valueOf(this.params.targetTimespan))).compareTo(this.params.proofOfWorkLimit) > 0) {
            log.info("Difficulty hit proof of work limit: {}", (Object)newDifficulty.toString(16));
            newDifficulty = this.params.proofOfWorkLimit;
        }
        int accuracyBytes = (int)(nextBlock.getDifficultyTarget() >>> 24) - 3;
        BigInteger receivedDifficulty = nextBlock.getDifficultyTargetAsInteger();
        BigInteger mask = BigInteger.valueOf(0xFFFFFFL).shiftLeft(accuracyBytes * 8);
        if ((newDifficulty = newDifficulty.and(mask)).compareTo(receivedDifficulty) != 0) {
            throw new VerificationException("Network provided difficulty bits do not match what was calculated: " + receivedDifficulty.toString(16) + " vs " + newDifficulty.toString(16));
        }
    }

    private void checkTestnetDifficulty(StoredBlock storedPrev, Block prev, Block next) throws VerificationException, BlockStoreException {
        Preconditions.checkState((boolean)this.lock.isLocked());
        long timeDelta = next.getTimeSeconds() - prev.getTimeSeconds();
        if (timeDelta >= 0L && timeDelta <= 1200L) {
            BigInteger newDifficulty;
            StoredBlock cursor = storedPrev;
            while (!cursor.getHeader().equals(this.params.genesisBlock) && cursor.getHeight() % this.params.interval != 0 && cursor.getHeader().getDifficultyTargetAsInteger().equals(this.params.proofOfWorkLimit)) {
                cursor = cursor.getPrev(this.blockStore);
            }
            BigInteger cursorDifficulty = cursor.getHeader().getDifficultyTargetAsInteger();
            if (!cursorDifficulty.equals(newDifficulty = next.getDifficultyTargetAsInteger())) {
                throw new VerificationException("Testnet block transition that is not allowed: " + Long.toHexString(cursor.getHeader().getDifficultyTarget()) + " vs " + Long.toHexString(next.getDifficultyTarget()));
            }
        }
    }

    private boolean containsRelevantTransactions(Block block) {
        for (Transaction tx : block.transactions) {
            try {
                for (BlockChainListener listener : this.listeners) {
                    if (!listener.isTransactionRelevant(tx)) continue;
                    return true;
                }
            }
            catch (ScriptException e) {
                log.warn("Failed to parse a script: " + e.toString());
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public StoredBlock getChainHead() {
        Object object = this.chainHeadLock;
        synchronized (object) {
            return this.chainHead;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Block getOrphanRoot(Sha256Hash from) {
        this.lock.lock();
        try {
            OrphanBlock tmp;
            OrphanBlock cursor = this.orphanBlocks.get(from);
            if (cursor == null) {
                Block block = null;
                return block;
            }
            while ((tmp = this.orphanBlocks.get(cursor.block.getPrevBlockHash())) != null) {
                cursor = tmp;
            }
            Block block = cursor.block;
            return block;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isOrphan(Sha256Hash block) {
        this.lock.lock();
        try {
            boolean bl = this.orphanBlocks.containsKey(block);
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Date estimateBlockTime(int height) {
        Object object = this.chainHeadLock;
        synchronized (object) {
            long offset = height - this.chainHead.getHeight();
            long headTime = this.chainHead.getHeader().getTimeSeconds();
            long estimated = headTime * 1000L + 600000L * offset;
            return new Date(estimated);
        }
    }

    public static enum NewBlockType {
        BEST_CHAIN,
        SIDE_CHAIN;

    }

    protected static class OrphanBlock {
        Block block;
        Set<Sha256Hash> filteredTxHashes;
        List<Transaction> filteredTxn;

        OrphanBlock(Block block, Set<Sha256Hash> filteredTxHashes, List<Transaction> filteredTxn) {
            Preconditions.checkArgument((block.transactions == null && filteredTxHashes != null && filteredTxn != null || block.transactions != null && filteredTxHashes == null && filteredTxn == null ? 1 : 0) != 0);
            this.block = block;
            this.filteredTxHashes = filteredTxHashes;
            this.filteredTxn = filteredTxn;
        }
    }
}

