/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.AddressMessage;
import com.google.bitcoin.core.AlertMessage;
import com.google.bitcoin.core.Block;
import com.google.bitcoin.core.BloomFilter;
import com.google.bitcoin.core.FilteredBlock;
import com.google.bitcoin.core.GetAddrMessage;
import com.google.bitcoin.core.GetBlocksMessage;
import com.google.bitcoin.core.GetDataMessage;
import com.google.bitcoin.core.GetHeadersMessage;
import com.google.bitcoin.core.HeadersMessage;
import com.google.bitcoin.core.InventoryMessage;
import com.google.bitcoin.core.MemoryPoolMessage;
import com.google.bitcoin.core.Message;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.NotFoundMessage;
import com.google.bitcoin.core.Ping;
import com.google.bitcoin.core.Pong;
import com.google.bitcoin.core.ProtocolException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.UnknownMessage;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VersionAck;
import com.google.bitcoin.core.VersionMessage;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BitcoinSerializer {
    private static final Logger log = LoggerFactory.getLogger(BitcoinSerializer.class);
    private static final int COMMAND_LEN = 12;
    private NetworkParameters params;
    private boolean parseLazy = false;
    private boolean parseRetain = false;
    private static Map<Class<? extends Message>, String> names = new HashMap<Class<? extends Message>, String>();

    public BitcoinSerializer(NetworkParameters params) {
        this(params, false, false);
    }

    public BitcoinSerializer(NetworkParameters params, boolean parseLazy, boolean parseRetain) {
        this.params = params;
        this.parseLazy = parseLazy;
        this.parseRetain = parseRetain;
    }

    public void serialize(Message message, OutputStream out) throws IOException {
        String name = names.get(message.getClass());
        if (name == null) {
            throw new Error("BitcoinSerializer doesn't currently know how to serialize " + message.getClass());
        }
        byte[] header = new byte[24];
        Utils.uint32ToByteArrayBE(this.params.packetMagic, header, 0);
        for (int i = 0; i < name.length() && i < 12; ++i) {
            header[4 + i] = (byte)(name.codePointAt(i) & 0xFF);
        }
        byte[] payload = message.bitcoinSerialize();
        Utils.uint32ToByteArrayLE(payload.length, header, 16);
        byte[] checksum = message.getChecksum();
        if (checksum == null) {
            Sha256Hash msgHash = message.getHash();
            if (msgHash != null && message instanceof Transaction) {
                int start;
                byte[] hash = msgHash.getBytes();
                for (int i = start = 20; i < start + 4; ++i) {
                    header[i] = hash[31 - i + start];
                }
            } else {
                byte[] hash = Utils.doubleDigest(payload);
                System.arraycopy(hash, 0, header, 20, 4);
            }
        } else {
            System.arraycopy(checksum, 0, header, 20, 4);
        }
        out.write(header);
        out.write(payload);
        if (log.isDebugEnabled()) {
            log.debug("Sending {} message: {}", (Object)name, (Object)(Utils.bytesToHexString(header) + Utils.bytesToHexString(payload)));
        }
    }

    public Message deserialize(InputStream in) throws ProtocolException, IOException {
        this.seekPastMagicBytes(in);
        BitcoinPacketHeader header = new BitcoinPacketHeader(in);
        return this.deserializePayload(header, in);
    }

    public BitcoinPacketHeader deserializeHeader(InputStream in) throws ProtocolException, IOException {
        return new BitcoinPacketHeader(in);
    }

    public Message deserializePayload(BitcoinPacketHeader header, InputStream in) throws ProtocolException, IOException {
        int bytesRead;
        byte[] payloadBytes = new byte[header.size];
        for (int readCursor = 0; readCursor < payloadBytes.length - 1; readCursor += bytesRead) {
            bytesRead = in.read(payloadBytes, readCursor, header.size - readCursor);
            if (bytesRead != -1) continue;
            throw new IOException("Socket is disconnected");
        }
        byte[] hash = Utils.doubleDigest(payloadBytes);
        if (header.checksum[0] != hash[0] || header.checksum[1] != hash[1] || header.checksum[2] != hash[2] || header.checksum[3] != hash[3]) {
            throw new ProtocolException("Checksum failed to verify, actual " + Utils.bytesToHexString(hash) + " vs " + Utils.bytesToHexString(header.checksum));
        }
        if (log.isDebugEnabled()) {
            log.debug("Received {} byte '{}' message: {}", new Object[]{header.size, header.command, Utils.bytesToHexString(payloadBytes)});
        }
        try {
            return this.makeMessage(header.command, header.size, payloadBytes, hash, header.checksum);
        }
        catch (Exception e) {
            throw new ProtocolException("Error deserializing message " + Utils.bytesToHexString(payloadBytes) + "\n", e);
        }
    }

    private Message makeMessage(String command, int length, byte[] payloadBytes, byte[] hash, byte[] checksum) throws ProtocolException {
        Message message;
        if (command.equals("version")) {
            return new VersionMessage(this.params, payloadBytes);
        }
        if (command.equals("inv")) {
            message = new InventoryMessage(this.params, payloadBytes, this.parseLazy, this.parseRetain, length);
        } else if (command.equals("block")) {
            message = new Block(this.params, payloadBytes, this.parseLazy, this.parseRetain, length);
        } else if (command.equals("merkleblock")) {
            message = new FilteredBlock(this.params, payloadBytes);
        } else if (command.equals("getdata")) {
            message = new GetDataMessage(this.params, payloadBytes, this.parseLazy, this.parseRetain, length);
        } else if (command.equals("tx")) {
            Transaction tx = new Transaction(this.params, payloadBytes, null, this.parseLazy, this.parseRetain, length);
            if (hash != null) {
                tx.setHash(new Sha256Hash(Utils.reverseBytes(hash)));
            }
            message = tx;
        } else if (command.equals("addr")) {
            message = new AddressMessage(this.params, payloadBytes, this.parseLazy, this.parseRetain, length);
        } else if (command.equals("ping")) {
            message = new Ping(this.params, payloadBytes);
        } else if (command.equals("pong")) {
            message = new Pong(this.params, payloadBytes);
        } else {
            if (command.equals("verack")) {
                return new VersionAck(this.params, payloadBytes);
            }
            if (command.equals("headers")) {
                return new HeadersMessage(this.params, payloadBytes);
            }
            if (command.equals("alert")) {
                return new AlertMessage(this.params, payloadBytes);
            }
            if (command.equals("filterload")) {
                return new BloomFilter(this.params, payloadBytes);
            }
            if (command.equals("notfound")) {
                return new NotFoundMessage(this.params, payloadBytes);
            }
            if (command.equals("mempool")) {
                return new MemoryPoolMessage();
            }
            log.warn("No support for deserializing message with name {}", (Object)command);
            return new UnknownMessage(this.params, command, payloadBytes);
        }
        if (checksum != null) {
            message.setChecksum(checksum);
        }
        return message;
    }

    public void seekPastMagicBytes(InputStream in) throws IOException {
        int magicCursor = 3;
        while (true) {
            int b;
            if ((b = in.read()) == -1) {
                throw new IOException("Socket is disconnected");
            }
            int expectedByte = 0xFF & (int)(this.params.packetMagic >>> magicCursor * 8);
            if (b == expectedByte) {
                if (--magicCursor >= 0) continue;
                return;
            }
            magicCursor = 3;
        }
    }

    public boolean isParseLazyMode() {
        return this.parseLazy;
    }

    public boolean isParseRetainMode() {
        return this.parseRetain;
    }

    static {
        names.put(VersionMessage.class, "version");
        names.put(InventoryMessage.class, "inv");
        names.put(Block.class, "block");
        names.put(GetDataMessage.class, "getdata");
        names.put(Transaction.class, "tx");
        names.put(AddressMessage.class, "addr");
        names.put(Ping.class, "ping");
        names.put(Pong.class, "pong");
        names.put(VersionAck.class, "verack");
        names.put(GetBlocksMessage.class, "getblocks");
        names.put(GetHeadersMessage.class, "getheaders");
        names.put(GetAddrMessage.class, "getaddr");
        names.put(HeadersMessage.class, "headers");
        names.put(BloomFilter.class, "filterload");
        names.put(FilteredBlock.class, "merkleblock");
        names.put(NotFoundMessage.class, "notfound");
        names.put(MemoryPoolMessage.class, "mempool");
    }

    public static class BitcoinPacketHeader {
        public final byte[] header = new byte[20];
        public final String command;
        public final int size;
        public final byte[] checksum;

        public BitcoinPacketHeader(InputStream in) throws ProtocolException, IOException {
            int cursor;
            int bytesRead;
            for (int readCursor = 0; readCursor < this.header.length; readCursor += bytesRead) {
                bytesRead = in.read(this.header, readCursor, this.header.length - readCursor);
                if (bytesRead != -1) continue;
                throw new IOException("Incomplete packet in underlying stream");
            }
            int mark = cursor = 0;
            while (this.header[cursor] != 0 && cursor - mark < 12) {
                ++cursor;
            }
            byte[] commandBytes = new byte[cursor - mark];
            System.arraycopy(this.header, mark, commandBytes, 0, cursor - mark);
            try {
                this.command = new String(commandBytes, "US-ASCII");
            }
            catch (UnsupportedEncodingException e) {
                throw new RuntimeException(e);
            }
            cursor = mark + 12;
            this.size = (int)Utils.readUint32(this.header, cursor);
            cursor += 4;
            if (this.size > 0x2000000) {
                throw new ProtocolException("Message size too large: " + this.size);
            }
            this.checksum = new byte[4];
            System.arraycopy(this.header, cursor, this.checksum, 0, 4);
            cursor += 4;
        }
    }
}

