(ns warpcore.db.core
  (:require [warpcore.util.core :as util]
            [datomic.api :as d]
            [taoensso.timbre :as timbre]
            [environ.core :refer [env]]))

(timbre/refer-timbre)

(def ^:dynamic *db-uri* (env :datomic-uri))

(defn conn [] (d/connect *db-uri*))

(defn db [] (d/db (conn)))

;; route helper
(defn wrap-db
  "A Ring middleware that provides a request-consistent database connection and
  value for the life of a request."
  [handler]
  (fn [request]
    (let [conn (d/connect *db-uri*)]
      (handler (assoc request
                 :conn conn
                 :db (d/db conn))))))

;; creation
(defn load-dtm [conn filename]
  (info "Loading schema file: " filename)
  ;; parse and submit schema transaction
  @(d/transact conn (read-string (slurp (clojure.java.io/resource filename)))))

(defn create! []
  ;; create database
  (info "Creating database" *db-uri*)
  (d/create-database *db-uri*)
  (let [conn (conn)] (load-dtm conn "db/schema.dtm") conn))

;; models

(def meta-props [:uuid :created-at :link-id])

(defn sanitize [props allowed-props & [disallowed-props]]
  (apply dissoc (select-keys props allowed-props)
         (concat meta-props disallowed-props)))

(defn save-and-return! [conn entity whitelist-props &
                        [{:keys [fill-attrs pull-props blacklist-props] :as opts}]]
  (debug "Creating entity:" entity)
  (let [{:keys [created-at? uuid?]} fill-attrs
        id (d/tempid :db.part/user) ;; revisit this. could get duplicated.
        {tempids :tempids dba :db-after}
        @(d/transact conn
                     [(merge (sanitize entity whitelist-props
                                       (concat meta-props blacklist-props))
                             (util/remove-nil-vals
                              {:db/id id
                               :uuid (if uuid? (d/squuid))
                               :created-at (if created-at? (java.util.Date.))}))])]
    (d/pull dba (or pull-props "[*]") (d/resolve-tempid (d/db conn) tempids id))))
