(ns warpcore.util.core
  (:require [clj-time.core :as dt]
            [clj-time.coerce :as dtc]
            [clj-time.format :as dtf]))

(defn remove-nil-vals [m]
  (apply dissoc m (for [[k v] m :when (nil? v)] k)))

(defn nil-safe [f]
  (fn [x & args]
    (if x (apply f x args))))

(defn new-uuid []
  (str (java.util.UUID/randomUUID)))

(defn pluralize [noun count]
  (if (= count 1)
    noun
    (str noun "s")))

;; time-ago
(defn make-phrase [i f unit]
  (let [n (f i)] (str n " " (pluralize unit n))))

(defn time-phrase [i]
  (cond
    (< (dt/in-millis i) 1000)
    (make-phrase i dt/in-millis "millisecond")
    (< (dt/in-seconds i) 60)
    (make-phrase i dt/in-seconds "second")
    (< (dt/in-minutes i) 60)
    (make-phrase i dt/in-minutes "minute")
    (< (dt/in-hours i) 24)
    (make-phrase i dt/in-hours "hour")
    (< (dt/in-days i) 7)
    (make-phrase i dt/in-days "day")
    (< (dt/in-days i) 31)
    (make-phrase i dt/in-weeks "week")
    (< (dt/in-years i) 1)
    (make-phrase i dt/in-months "month")
    :default
    (make-phrase i dt/in-years "year")))

(defn days-ago
  [ts]
  (let [now (dt/now)]
    (when (dt/after? now ts)
      (dt/in-days (dt/interval ts now)))))

(defn after-now?
  [ts]
  (dt/after? (dtc/from-date ts) (dt/now)))

(defn before-now?
  [ts]
  (dt/before? (dtc/from-date ts) (dt/now)))

(defn time-ago
  "Transforms timestamp into human-readable phrase, etc. '2 hours ago'.
   Returns nil if timestamp is in the future."
  [ts]
  (let [now (dt/now)] (when (dt/after? now ts) (time-phrase (dt/interval ts now)))))
