![Prismo](prismo.gif)

# prismo

Prismo is a Clojure interface to Couchbase. It wraps the Java API
provided by Couchbase.

Prismo can insert, replace ("upsert"), retrieve ("get"), and remove
documents. It works with JSON documents, string documents, and binary
documents. There is an asynchronous interface (using
[RxClojure](https://github.com/ReactiveX/RxClojure)/[RxJava](https://github.com/ReactiveX/RxJava)),
as well as synchronous convenience functions for inserting, getting,
or removing multiple documents at a time. There is support for
document expiration, timeouts, and operating on replicas.

Prismo is not complete. There is not yet any support for views, N1QL,
or administation.

## Installation

Add the following dependency to your `project.clj` file:

    `[com.yieldbot/prismo "0.3.0"]`

## Usage

### Synopsis

```clojure
user> (require '[prismo.core :as p])
nil
user> (def cluster (p/create-cluster ["couchbase-1.example.com"
                                      "couchbase-2.exampled.com"]))
#'user/cluster
user> (def bucket (p/open-bucket cluster "default"))
#'user/bucket
user> (p/insert bucket (p/make-raw-json-document "foo" {"pi" 3.14}))
#<RawJsonDocument RawJsonDocument{id='foo', cas=593401636448764, expiry=0, content={"pi":3.14}}>
user> (def doc (p/get bucket (p/make-raw-json-document "foo")))
#<RawJsonDocument RawJsonDocument{id='foo', cas=593401636448764, expiry=0, content={"pi":3.14}}>
user> (p/id doc)
"foo"
user> (p/content doc)
{"pi" 3.14}
user> (p/remove bucket (p/make-raw-json-document "foo"))
#<RawJsonDocument RawJsonDocument{id='foo', cas=593401636448765, expiry=0, content=null}>
```

### Making documents

As a convenience, `make-raw-json-document`, `make-string-document` and
`make-binary-document` will return `nil` if called with `nil` as the
only argument (instead of a valid key).


### Inserting, removing, or getting multiple documents at once

The functions `multi-insert`, `multi-get`, `multi-remove`, and
`multi-counter` submit multiple operations asyncronously to Couchbase,
then waits for them all to continue before returning.

All the `multi-…` functions take two arguments: a bucket, and a
sequence of documents (or, in the case of `multi-counter`, document
IDs). They return a sequence of results. There is a result for each
input document, and the results are in the same order as the input
documents. The results are typically documents but they can also be
`Exception` instances or `nil`, as explained in the following:

* If you specify `:on-error :collect` to any of the `multi-…`
  functions, then any exceptions that are thrown during the
  asyncronous operations will be caught, and the `Exception` instances
  will be included at the appropriate place in the results sequence.
  ```clojure
user> (p/multi-insert bucket [(p/make-raw-json-document "foo" 3.14)
                                  (p/make-raw-json-document "bar" 2.72)]
                                  :on-error :collect)
(#<DocumentAlreadyExistsException com.couchbase.client.java.error.DocumentAlreadyExistsException>
     #<RawJsonDocument RawJsonDocument{id='bar', cas=586974473324637, expiry=0, content=2.72}>)
```

* If you specify `:on-error :throw` (the default), an `ExceptionInfo`
  will be thrown if any of the operations throws an exception. In the
  exception data, `:results` maps to the sequence of results that
  would have been returned if you had specified `:on-error :collect`.
  ```clojure
user> (p/multi-insert bucket [(p/make-raw-json-document "foo" 3.14)
                                  (p/make-raw-json-document "bar" 2.72)])
ExceptionInfo Some :insert operations failed.  clojure.core/ex-info (core.clj:4577)
user> (:results (ex-data *e))
(#<DocumentAlreadyExistsException com.couchbase.client.java.error.DocumentAlreadyExistsException>
     #<RawJsonDocument RawJsonDocument{id='bar', cas=587239143980322, expiry=0, content=2.72}>)
```

* If you specify `:if-does-not-exist` to `multi-remove`, a missing
  document will lead to a `nil` in the results list:
  ```clojure
user> (p/multi-remove bucket [(p/make-raw-json-document "foo")
                                  (p/make-raw-json-document "bar")]
                                  :if-does-not-exist :ignore)
(#<RawJsonDocument RawJsonDocument{id='foo', cas=591891927054114, expiry=0, content=null}>
     nil)
     ```

The `multi-insert`, `multi-get`, `multi-remove`, and `multi-counter`
function take a `:timeout` keyword argument that allows you to specify
a number of milliseconds after which a
`java.util.concorrent.TimeoutException` should be thrown. (These
exceptions are collected or combined and thrown according to
`:on-error` as previously described.)

It is acceptable for the sequence of documents that you give as input
to the `multi-…` functions to contain `nil`. In that case, there will
be a `nil` at the corresponding place in the returned sequence.

The delta and initial value for `multi-counter` are always 1.


### Document expiration

The `make-*-document` functions take optional keyword argument
`:expiry` that allows you to set a TTL for the document you are about
to insert. The TTL is an integer, generally the absolute time (in unix
time) for the document to expire.
```clojure
user> (p/make-raw-json-document "foo" 3.14 :expiry 1441830282)
#<RawJsonDocument RawJsonDocument{id='foo', cas=0, expiry=1441830282, content=3.14}>
```

Alternately, you can give the number
of seconds that the document should live before expiring, but this
only works for 30 days or less.
```clojure
user> (p/make-raw-json-document "foo" 3.14 :expiry 3600)
#<RawJsonDocument RawJsonDocument{id='foo', cas=0, expiry=3600, content=3.14}>
```

The default is `0`, which means that the document will not expire.

At present, there is no way to specify expiration for `multi-counter`.


### RxJava interface

The `prismo.rx` contains an asynchronous interface that uses reactive
extensions (Rx). The function `prismo.rx/multi` issues insert, upsert,
remove, or get operations for a sequence of documents. It returns an
observable that emits vectors, each with two elements: (1) a document
or key from among the ones given as input to `multi` and (2) the
result of the operation. If the operation fails, the result will be an
exception instance.

```clojure
user> (require '[prismo.rx :as prx])
nil
user> (pprint (rx.lang.clojure.blocking/o->seq
               (prx/multi :get bucket [(p/make-raw-json-document "foo")]))
([#<RawJsonDocument RawJsonDocument{id='foo', cas=0, expiry=0, content=null}> 
  #<RawJsonDocument RawJsonDocument{id='foo', cas=16242619652794359, expiry=0, content=[42,3.14]}>])
```

Note that the vectors can be emitted in any order. There will not
necessarily be a vector emitted for every input document: if you try
to get a nonexistant document, the observable will complete without
emitting a vector for that document:

```clojure
user> (rx.lang.clojure.blocking/o->seq
       (prx/multi :get bucket [(p/make-raw-json-document "nonexistant")]))
nil
```

The `:timeout` keyword argument lets you specify a timeout in
milliseconds. Operations that do not complete in time will return a
`TimeoutException`:

```clojure
user> (pprint (rx.lang.clojure.blocking/o->seq
               (prx/multi :get bucket [(p/make-raw-json-document "foo")]
                          :timeout 1)))
([#<RawJsonDocument RawJsonDocument{id='foo', cas=0, expiry=0, content=null}>
  #<TimeoutException java.util.concurrent.TimeoutException>])
  ```

### Reading from replicas

If a server is down, you can pass `:replica :first`, `:replica
:second`, or `:replica :third` to `multi-get` in order to read from
one of the replicas.

At present, this is only supported for `multi-insert`, `multi-get`,
and `multi-remove`.

## License

Copyright © 2015 Yieldbot, Inc.

Distributed under the Eclipse Public License either version 1.0 or (at
your option) any later version.
