(ns yunjia.util.middleware
  (:require [ring.util.http-response :as hr]))

(defn wrap-authentication
  "处理认证的Ring中间件，返回Ring的handler。
  放行已认证的请求，拦截未认证的请求。
  通过session实现。在session中增加:auth-access-time，记录最后一次已通过认证请求的访问时间。
  seconds: session过期的秒数。
  auth-fail-handler: 如未通过认证，由该处理器产生响应。若无此参数，仅返回403响应。
  "
  [handler seconds & [auth-fail-handler]]
  (fn [request]
    (let [access-time (get-in request [:session :auth-access-time])
          current-time (System/currentTimeMillis)
          auth-ok (and access-time
                       (> (- current-time access-time) 0)
                       (< (- current-time access-time) (* seconds 1000)))
          fail-handler (if auth-fail-handler
                         auth-fail-handler
                         (fn [r] (hr/forbidden)))]
      (if auth-ok
        ; 验证通过，放行，并更新session
        (let [response (handler request)
              not-found (= (str (:status response))
                           "404")]
          (if not-found
            response
            (update response :session assoc :auth-access-time current-time)))
        ; 验证未通过，需要删除session
        (-> request
            (assoc :session nil)
            fail-handler
            (assoc :session nil))))))




