(ns yunjia.util.time
  (:require [yunjia.util.string :as yj-string])
  (:import [java.time Instant LocalDate LocalDateTime LocalTime Period ZonedDateTime ZoneId YearMonth]
           java.time.format.DateTimeFormatter
           java.util.Calendar
           (java.time.temporal ChronoUnit)))

(def ^ZoneId zone-shanghai
  (ZoneId/of "Asia/Shanghai"))

(def formatter-yyyyMMddHHmm (DateTimeFormatter/ofPattern "yyyyMMddHHmm"))

(def formatter-yyyyMMddHHmmss (DateTimeFormatter/ofPattern "yyyyMMddHHmmss"))

(defn ^String now-yyyyMMddHHmm
  []
  (let [dt (ZonedDateTime/ofInstant (Instant/now) zone-shanghai)]
    (.format dt formatter-yyyyMMddHHmm)))

(defn ^String now-yyyyMMddHHmmss
  []
  (let [dt (ZonedDateTime/ofInstant (Instant/now) zone-shanghai)]
    (.format dt formatter-yyyyMMddHHmmss)))

(defn ^String timestamp->format
  "格式化时间戳。"
  [timestamp
   ^String format-string]
  (if timestamp
    (let [dt (ZonedDateTime/ofInstant (Instant/ofEpochSecond timestamp) zone-shanghai)
          formatter (DateTimeFormatter/ofPattern format-string)]
      (.format dt formatter))
    ""))

(defn calc-years
  "计算某个时间戳到当前时间的年数，不足整年部分也算一年。"
  [^long epoch-second]
  (let [instant (Instant/ofEpochSecond epoch-second)
        datetime (LocalDateTime/ofInstant instant zone-shanghai)
        date (.toLocalDate datetime)
        today (.toLocalDate (LocalDateTime/now zone-shanghai))
        period (Period/between date today)]
    (.getYears period)))

(defn to-timestamp
  [year month day & [hour minute second]]
  (let [hour (if hour hour 0)
        minute (if minute minute 0)
        second (if second second 0)
        local-date (LocalDate/of ^long year ^long month ^long day)
        local-time (LocalTime/of hour minute second)]
    (-> (ZonedDateTime/of local-date local-time zone-shanghai)
        .toEpochSecond)))

(defn yyyyMMdd->timestamp
  "默认日期格式为'yyyyMMdd'转换成时间戳
  [^String date-string    ^String format]可选日期格式转换成时间戳"
  ([^String day-string] (yyyyMMdd->timestamp day-string "00" "00" "00"))
  ([^String day-string
    ^String hour-string] (yyyyMMdd->timestamp day-string hour-string "00" "00"))
  ([^String day-string
    ^String hour-string
    ^String minute-string] (yyyyMMdd->timestamp day-string hour-string minute-string "00"))
  ([^String day-string
    ^String hour-string
    ^String minute-string
    ^String second-string]
   (let [formatter (DateTimeFormatter/ofPattern "yyyyMMdd")
         local-date (LocalDate/parse day-string formatter)
         local-time (LocalTime/of (Integer/parseInt hour-string) (Integer/parseInt minute-string) (Integer/parseInt second-string))]
     (-> (ZonedDateTime/of local-date local-time zone-shanghai)
         .toEpochSecond))))

(defn timestamp->datatime
  "时间戳解析成包括各单位的map"
  [^long timestamp]
  (let [instant (Instant/ofEpochSecond timestamp)
        zoned-data-time (ZonedDateTime/ofInstant instant zone-shanghai)]
    {:year   (.getYear zoned-data-time)
     :month  (.getMonthValue zoned-data-time)
     :day    (.getDayOfMonth zoned-data-time)
     :hour   (.getHour zoned-data-time)
     :minute (.getMinute zoned-data-time)
     :second (.getSecond zoned-data-time)}))

(defn begin-day-of-month-in-millis
  "获取本月开始时间戳in millis"
  []
  (let [calender (Calendar/getInstance)]
    (.set calender
          (.get calender Calendar/YEAR)
          (.get calender Calendar/MONDAY)
          (.get calender Calendar/DAY_OF_MONTH)
          0
          0
          0)
    (.set calender
          Calendar/DAY_OF_MONTH
          (.getActualMinimum calender Calendar/DAY_OF_MONTH))
    (.getTimeInMillis calender)))

(defn end-day-of-month-in-millis
  "获取本月结束时间戳in millis"
  []
  (let [calender (Calendar/getInstance)]
    (.set calender
          (.get calender Calendar/YEAR)
          (.get calender Calendar/MONDAY)
          (.get calender Calendar/DAY_OF_MONTH)
          0
          0
          0)
    (.set calender
          Calendar/DAY_OF_MONTH
          (.getActualMaximum calender Calendar/DAY_OF_MONTH))
    (.set calender
          Calendar/HOUR_OF_DAY
          24)
    (.getTimeInMillis calender)))

(defn begin-day-of-week-in-millis
  "获取本周开始时间戳in millis"
  []
  (let [calender (Calendar/getInstance)]
    (.set calender
          (.get calender Calendar/YEAR)
          (.get calender Calendar/MONDAY)
          (.get calender Calendar/DAY_OF_MONTH)
          0
          0
          0)
    (.set calender
          Calendar/DAY_OF_WEEK
          Calendar/MONDAY)
    (.getTimeInMillis calender)))

(defn end-day-of-week-in-millis
  "获取本周结束时间戳in millis"
  []
  (let [calender (Calendar/getInstance)]
    (.setTimeInMillis calender (begin-day-of-week-in-millis))
    (.add calender
          Calendar/DAY_OF_WEEK
          7)
    (.getTimeInMillis calender)))

(defn yyyyMM->begin&end-month
  "获取本月开始时间戳in millis"
  [& [month-inside-day]]
  (let [year&month (if month-inside-day
                     (-> (yyyyMMdd->timestamp month-inside-day)
                         (timestamp->format "yyyyMM"))
                     (-> (System/currentTimeMillis)
                         (quot 1000)
                         (timestamp->format "yyyyMM")))
        this-begin-day (str year&month "01")

        formatter (DateTimeFormatter/ofPattern "yyyyMMdd")
        local-date (LocalDate/parse this-begin-day formatter)
        local-time (LocalTime/of 0 0 0)

        this-begin-timestamp (-> (ZonedDateTime/of local-date local-time zone-shanghai)
                                 .toEpochSecond)

        ;;前一个月份的开始时间
        previous-month-begin (.minus local-date 1 ChronoUnit/MONTHS)

        previous-begin-timestamp (-> (ZonedDateTime/of previous-month-begin local-time zone-shanghai)
                                 .toEpochSecond)

        ;;下一个月份的开始时间
        next-month-begin (.plus local-date 1 ChronoUnit/MONTHS)

        next-begin-timestamp (-> (ZonedDateTime/of next-month-begin local-time zone-shanghai)
                                 .toEpochSecond)]
    [previous-begin-timestamp this-begin-timestamp next-begin-timestamp]))
