(ns yunjia.util.jsonapi
  "用于处理jsonapi规范。请参考http://jsonapi.org/。"
  (:require [ring.util.response :as resp]
            [cheshire.core :as json]
            [ring.util.request :refer [request-url]]
            [clojure.walk :refer [keywordize-keys]]
            [schema.core :as s]
            [medley.core :refer [map-vals]]
            [clj-http.client :as client]))

;;
;; 定义schema
;;

(def AnyMap {s/Any s/Any})

(def StrOrKey (s/cond-pre s/Str s/Keyword))

(def StrOrMap (s/cond-pre s/Str AnyMap))

(def Attributes
  {StrOrKey s/Any})

(def Links
  {StrOrKey StrOrMap})

(def Relationship
  {(s/optional-key :links) Links
   (s/optional-key :data)  s/Any
   (s/optional-key :meta)  s/Any})

(def Relationships
  {StrOrKey Relationship})

(def ResourceIdentifier
  {:id   StrOrKey
   :type StrOrKey})

(def Resource
  {:id                             StrOrKey
   :type                           StrOrKey
   (s/optional-key :attributes)    Attributes
   (s/optional-key :links)         Links
   (s/optional-key :relationships) Relationships})

(def PrimaryData
  (s/cond-pre [Resource] (s/maybe Resource)))

;;
;; 定义jsonapi操作
;;

(def content-type "application/vnd.api+json; charset=utf-8")

(defn document
  "创建包含主数据的jsonapi文档"
  ([data] {:data data})
  ([data links] {:data data, :links links})
  ([data links included]
   {:data     data
    :links    links
    :included included}))

(defn error-document
  "创建包含错误信息的jsonapi文档。"
  [error]
  ;; TODO
  )

(s/defn links :- Links
  "创建顶层links对象。request之后，接受偶数个参数，形如: name1 link1 name2 link2 ..."
  [request & name-link-seq]
  (let [links {:self (request-url request)}]
    (if name-link-seq
      (keywordize-keys (apply assoc links name-link-seq))
      links)))

(defn resource-identifier
  "创建资源标识。"
  [id type]
  {:id (str id), :type (name type)})

(defn resource
  "创建资源。"
  ([id type] (resource-identifier id type))
  ([id type attributes] (assoc
                          (resource-identifier id type)
                          :attributes attributes))
  ([id type attributes relationships] (assoc
                                        (resource id type attributes)
                                        :relationships relationships))
  ([id type attributes relationships links] (assoc
                                              (resource id type attributes relationships)
                                              :links links)))

(defn relationship?
  "判断是否relationship对象。"
  [relationship]
  (and (associative? relationship)
       (some #{:links :data :meta} (keys relationship))))

(defn relationships
  "创建relationships。接受偶数个参数，形如: name1 relationship1 name2 relationship2 ..."
  [name relationship & nrs]
  (let [relationships (if nrs
                        (apply assoc {name relationship} nrs)
                        {name relationship})]
    {:relationships (keywordize-keys relationships)}))

(defn json-response
  "将body转换为json字符串"
  [response]
  (update-in response [:body] (fn [body]
                                (cond
                                  (nil? body) body
                                  (string? body) body
                                  :else (json/encode body)))))

(defn response
  "处理响应map：
  1、增加content-type
  2、如果(associative? body)为真，将body转换为json字符串"
  [response]
  (->
    response
    (resp/content-type content-type)
    json-response))

;;
;; 一些有用的工具函数
;;

(defn resource-select-keys
  "从资源map中选择一些元素，返回一个map。如果资源中不存在某个key，则返回map中不包含该key。
  该函数调用方式与select-keys类似。"
  [resource ks]
  (if resource
    (let [m1 (select-keys (select-keys resource [:id :type]) ks)
          m2 (select-keys (:attributes resource) ks)
          r (select-keys (:relationships resource) ks)
          m3 (map-vals #(get-in % [:data :id]) r)]
      (merge m1 m2 m3))
    {}))

(defn- process-client-response
  "客户请求jsonapi服务，对得到的响应进行处理，如json解码等。"
  [response]
  (update response :body
          #(if % (json/decode % true))))

(defn client-get
  "clj-http.client/get的包装函数，调用方式和原函数一致。
  对请求做了一些处理，增加了content-type等。
  对响应的body进行json解码。"
  [url & [req]]
  (let [resp (client/get url
                         (merge {:content-type content-type} req))]
    (process-client-response resp)))

(defn client-post
  "clj-http.client/post的包装函数，调用方式和原函数一致。
  对请求做了一些处理，增加了content-type等。
  对响应的body进行json解码。"
  [url & [req]]
  (let [resp (client/post url
                          (merge {:content-type content-type} req))]
    (process-client-response resp)))

(defn map-to-resource
  "将map（例如从数据库查询得到的）转换为resource，转换结果不包含relationships和links。"
  [m type & [id-key]]
  (let [idk (or id-key
                :id)
        attributes (dissoc m idk)]
    (resource (idk m) type attributes)))

(defn to-resource-seq
  "将map序列（例如从数据库查询得到的）转换为resource序列，转换结果不包含relationships和links。"
  [map-seq type & [id-key]]
  (let [idk (or id-key
                :id)]
    (map #(map-to-resource % type idk) map-seq)))