(ns yunjia.util.component.redis-client
  "使用jedis访问redis
  create by yan 2018/01/02 15:37"
  (:require [com.stuartsierra.component :as component]
            [taoensso.timbre :as timbre])
  (:import (java.util HashSet)
           (redis.clients.jedis HostAndPort JedisCluster Jedis)
           (org.apache.commons.pool2.impl GenericObjectPoolConfig)))

(defrecord redis-client [config-redis-key config]
  component/Lifecycle

  (start [this]
    (timbre/debug "Starting redis-client component...")
    (if (:redis-client this)
      this
      (let [redis-key (if config-redis-key
                        (keyword config-redis-key)
                        :redis-config)
            redis-config (get-in config [:value redis-key])
            {:keys [nodes host port password
                    connection-timeout so-timeout max-attempts]} redis-config

            ;;连接Redis Server超时时间
            connection-timeout (-> (or connection-timeout 2000)
                                   (str)
                                   (Integer/parseInt))

            ;;等待Response超时时间
            so-timeout (-> (or so-timeout 2000)
                           (str)
                           (Integer/parseInt))

            ;;设置最大连接数
            max-attempts (-> (or so-timeout 5)
                             (str)
                             (Integer/parseInt))

            ;;非集群部署
            redis-host (or host "127.0.0.1")
            redis-port (-> (or port 6379)
                           (str)
                           (Integer/parseInt))

            ;;集群部署
            ;;new 一个cluster 节点set
            jedis-cluster-nodes (new HashSet)

            ;;添加节点的host、port
            _ (doseq [{:keys [host ports]} nodes]
                (doseq [port ports]
                  (let [node-host (str (or host "127.0.0.1"))
                        node-port (-> (or port 6379)
                                      (str)
                                      (Integer/parseInt))]
                    (->> (new HostAndPort node-host node-port)
                         (.add jedis-cluster-nodes)))))

            ;Commons Pool的参数
            pool-config (new GenericObjectPoolConfig)

            ;;优先选择连接集群
            jedis-client (cond
                           (.isEmpty jedis-cluster-nodes) (let [jedis-client-ex (new Jedis redis-host redis-port connection-timeout so-timeout)]
                                                            (if password
                                                              (.auth jedis-client-ex password))
                                                            jedis-client-ex)
                           password (new JedisCluster jedis-cluster-nodes connection-timeout so-timeout max-attempts password pool-config)
                           :else (new JedisCluster jedis-cluster-nodes connection-timeout so-timeout max-attempts pool-config))]

        (assoc this :redis-client jedis-client))))

  (stop [this]
    (timbre/debug "Stopping redis-client component...")
    ;(if (:redis_client this)
    ;  ;(component/stop (:redis_client this))
    ;  )
    (assoc this :redis-client nil)))

(defn make-redis-client
  ([] (make-redis-client :redis-config))
  ([config-redis-key] (map->redis-client {:config-redis-key (keyword config-redis-key)})))