# Carousel

Content carousel component for Reagent (demo: https://aw7.github.io/z-com-demo/#carousel).

## Installation

*To install the entire library see https://github.com/aw7/z-com.*

* add the dependency:
  ```Clojure
  :dependencies [[z-com/carousel "0.4.0"]]
  ````

* include the stylesheet `resources/public/css/carousel.css` in your HTML

* if you want to use ZMDI's chevron icons for the arrows include the [ZMDI stylesheet](https://cdnjs.cloudflare.com/ajax/libs/material-design-iconic-font/2.1.2/css/material-design-iconic-font.min.css) as well


## Usage

There are three ways to specify a *carousel* depending on the representation of the contents you want to navigate.

* `:item-seq` a sequence of Reagent components:

  ```Clojure
  (ns carousel-app
    (:require [reagent.core :as r]
              [z-com.core :as z]))

  (defn index []
    [:div [:h1 "Photos"]
      [z/carousel :item-seq [[:img {:src "photo1.jpg"}]
                             [:img {:src "photo2.jpg"}]]]])
  ```

* `:item-fn` a function of the active index returning a Reagent component:

  ```Clojure
  (def photos ["photo1.jpg" "photo2.jpg"])

  (defn index []
    [:div [:h1 "Photos"]
      [z/carousel :item-fn (fn [idx] [:img {:src (nth photos idx)}])
                  :item-cnt (count photos)]])
  ```

  Since *carousel* can't count the elements as before to determine the highest valid index it has to be supplied through `:item-cnt`.

* `:item` a Reagent component:

  ```Clojure
  (defn index []
    (r/with-let [idx-ratom (r/atom 0)
                 photos [[:img {:src "photo1.jpg"}]
                         [:img {:src "photo2.jpg"}]]]
      [:div [:h1 "Photos"]
        [z/carousel :item [:img {:src (nth photos @idx-ratom)}]
                    :prev-fn (swap! idx-ratom dec)
                    :next-fn (swap! idx-ratom inc)]]))
  ```

  Content is provided directly as a Reagent component and clients have to manage navigation state externally by supplying prev/next handlers.


### Optional parameters

* `:idx-model`

  To track the currently active item *carousel* uses an index r/atom, which is either instantiated internally (in connection with `:item-seq` and `:item-fn`) or supplied manually, allowing clients to incorporate that index into a status indicator and/or control the index externally:

  ```Clojure
  (defn index []
    (r/with-let [idx-ratom (r/atom 0)
                 photos [[:img {:src "photo1.jpg"}]
                         [:img {:src "photo2.jpg"}]]]
      [:div [:h1 "Photos" [:span.badge (str (inc @idx-ratom) " of " (count photos))]]
        [z/carousel :item-seq photos
                    :idx-model idx-ratom]]))
  ```

* `:slide?` CSS-based slide animation for transitions
* `:arrow-position` can be set to `:outside` (default) or `:inside`
* `:arrow-style` supports `:guillemets` (default) or `:zmdi`
* `:arrow-left` & `:arrow-right` can also be specified directly as an alternative to `:arrow-style`
* `:arrows-hidden?` if true the arrows are only shown when hovering the carousel (only supported in connection with `:arrow-position :inside`)
