(ns elastic-input.core
  (:require [reagent.core :as r]))

;; ----------------------------------------------------------------------------
;; scrolling

(defn scroll-l [dom-node] (set! (.-scrollLeft dom-node) 0))
(defn scroll-t [dom-node] (set! (.-scrollTop  dom-node) 0))
(defn tgt-val  [e]        (-> e .-target .-value))

(defn scroll-factory [component scroll-f]
  (fn [opts]
    (r/create-class
      {:reagent-render
        (fn [{:as   opts
              :keys [component-opts
                     text placeholder width height
                     on-text-change on-focus on-blur on-key-press]
              :or   {on-focus #()
                     on-blur  #()
                     on-key-press #()}}]
          [component
            (merge component-opts
              {:value        text
               :placeholder  placeholder
               :style        {:width width :height height}
               :on-focus     (comp on-focus       tgt-val)
               :on-change    (comp on-text-change tgt-val)
               :on-blur      (comp on-blur        tgt-val)
               :on-key-press (fn [e] (on-key-press (max (.-keyCode e)
                                                        (.-which e))))})])
       :component-did-update
         (comp scroll-f r/dom-node)})))

(def input    (scroll-factory :input    scroll-l))
(def textarea (scroll-factory :textarea scroll-t))



;; ----------------------------------------------------------------------------
;; focus-ctrl

(defn focus-ctrl-factory
  "Create a handler for component-did-mount and component-did-update"
  [focus-tracker reset-scroll]
  (fn [this]  
    (let [n (r/dom-node this)
          _ (reset-scroll n)
          {:keys [curr prev]} @focus-tracker]
      (cond (and (not prev) curr)
            (do (swap! focus-tracker #(assoc % :prev true))
                (.focus n))
            (and prev (not curr))
            (do (swap! focus-tracker #(assoc % :prev false))
                (.blur n))
            :else true))))

(defn focus-factory [component]
  (fn [opts]
    (let [focus-tracker (r/atom {:curr false :prev false})
          focus-ctrl    (focus-ctrl-factory focus-tracker scroll-l)]
      (r/create-class
        {:reagent-render
           (fn [{:as opts :keys [focused?]}]
             (when (not= focused? (:curr @focus-tracker))
               (swap! focus-tracker #(update-in % [:curr] not)))
             [component
               (update-in opts [:component-opts]
                 #(assoc % :class (if (:curr @focus-tracker) "active" "inactive")))])
         :component-did-update focus-ctrl
         :component-did-mount  focus-ctrl}))))

(def input-foc    (focus-factory input))
(def textarea-foc (focus-factory textarea))



;; ----------------------------------------------------------------------------
;; phantom-ctrl

(defn escape [txt]
  (clojure.string/escape txt
    {\< "&lt;"
     \> "&gt;"
     \& "&amp;"
     "\n" "<br>"}))

(defn phantom-ctrl-factory [inspect report display-style]
  (fn [this]
    (let [n (r/dom-node this)
          _ (-> n .-style .-display (set! display-style))
          h (inspect n)
          _ (-> n .-style .-display (set! "none"))]
     (report h))))

(defn phantom-factory [inspect report visible-style sentinel-str]
  (let [phantom-ctrl (phantom-ctrl-factory inspect report visible-style)]
    (fn [opts]
      (r/create-class
        {:reagent-render
         (fn [{:as opts :keys [text width height]}]
           [:div.phantom
             {:style {:width width :height height}
              :dangerouslySetInnerHTML
               {:__html (str (escape text) sentinel-str)}}])
         :component-did-update phantom-ctrl
         :component-did-mount  phantom-ctrl}))))

(defn auto-factory
  [{:keys [component-foc component
           min-len extra-len
           set-f visible-style sentinel-str
           dimension class-name]}]
  (fn [{:as opts :keys [on-text-change on-blur]}]
    (r/with-let
      [len       (r/atom min-len)
       min-len   (get opts :min-len   min-len)
       extra-len (get opts :extra-len extra-len)
       phantom   (phantom-factory
                   set-f
                   #(reset! len (+ (if (> % min-len) % min-len)
                                   extra-len))
                   visible-style
                   sentinel-str)
       buffer-ratom (when (contains? opts :buffer?) (r/atom (:text opts)))
       buffer-changed?-ratom (when (contains? opts :buffer?) (r/atom false))
       update-buffer
         #(do (reset! buffer-ratom %)
            (reset! buffer-changed?-ratom true))
       commit-buffer
         #(do (when @buffer-changed?-ratom
                (do (on-text-change @buffer-ratom)
                    (reset! buffer-changed?-ratom false)))
              (when on-blur (on-blur)))]
      (let [opts (if (:buffer? opts)
                   (merge opts
                     {:on-text-change update-buffer
                      :on-blur        commit-buffer
                      :on-key-press   (fn [key-code]
                                        (when (= 13 key-code) (commit-buffer)))
                      :text           @buffer-ratom})
                   opts)
            opts (assoc opts dimension @len)]
        [:div.elastic-input (:container opts)
          (if (contains? opts :focused?)
            [component-foc opts]
            [component     opts])
          (let [opts (update-in opts [:text]
                                #(if (= "" %) (or (:placeholder opts) "") %))]
            [phantom (select-keys opts (case dimension
                                         :height [:text :width]
                                         :width  [:text]))])]))))

(def input-auto-w (auto-factory
  {:component-foc input-foc
   :component     input
   :min-len       10
   :extra-len     5
   :set-f         #(.-clientWidth %)
   :visible-style "inline-block"
   :sentinel-str  ""
   :dimension     :width
   :class         "input-auto-w"}))

(def textarea-auto-h (auto-factory
  {:component-foc textarea-foc
   :component     textarea
   :min-len       25
   :extra-len     0
   :set-f         #(.-clientHeight %)
   :visible-style "block"
   :sentinel-str  "<br class='lbr'>"
   :dimension     :height
   :class         "textarea-auto-h"}))


(defn elastic-input
  [& {:as args :keys [width height]}]
  (cond (= width  :auto) [input-auto-w args]
        (= height :auto) [textarea-auto-h args]))
