(ns navtree.core
  (:require [reagent.core :as r]
            [reagent.ratom :refer [reaction]]
            [taoensso.timbre :as tb]
            [keybind.core :as kb]
            [clojure.zip :as z]
            [navtree.zipper :as zn]))

(def realized-ds? #(cond (nil? %) true
                         (satisfies? IPending %) (realized? %)
                         :else true))

(def realization  #(cond (delay? %) (-> % deref doall)
                         :else      (-> % doall)))

(defn -loader [hiccup data-list]
  (r/create-class
    {:reagent-render (fn [hiccup _] (or hiccup [:span.loading-indicator]))
     :component-did-mount
     (fn [_]
       (js/setTimeout
         #(doseq [{:keys [d-val d-ratom d-real?]} data-list]
            (realization d-val)
            (reset! d-real? true)
         10)))}))

(defn -navtree-node
  [{:as params
    :keys [selection-path on-selection-path whitelist
           id-fn label-fn children-fn content-fn content-ratom
           loading-indicator]}
   id-path ;; terminal-headed
   tree]
  (r/with-let [id      (id-fn tree)
               id-path (conj id-path id)

               cc (when children-fn (children-fn tree))
               d  (when content-fn  (content-fn id))

               children-realized? (r/atom (realized-ds? cc))
               content-realized?  (r/atom (realized-ds? d))]

    (let [sel? (= id (last selection-path))
          ex?  (cond whitelist  (contains? whitelist id)
                     :else true)

          data-realized?
          (cond-> true ex?  (and @children-realized?)
                       sel? (and @content-realized?))

          loader-args
          (cond-> (list)
            ex?  (conj {:d-val cc :d-real? children-realized?})
            sel? (conj {:d-val d  :d-real? content-realized?}))]

      ^{:key (hash id)}
      [:li
        {:class (str (when sel? "selection ")
                     (if ex? "expanded" "collapsed"))}
        ;; header
        [:span.header {:on-click (partial on-selection-path (reverse id-path))}
          (label-fn tree)]

        ;; loading indicator
        (if data-realized?
          (when sel?
            (do (reset! content-ratom (realization d)) nil))
          [-loader loading-indicator loader-args])

        ;; children
        (when (and ex? @children-realized?)
          [:ul.children
            (doall
              (for [c (realization cc)]
                ^{:key (hash (id-fn c))}
                [-navtree-node params id-path c]))])])))

(def handle-opt-delay #(if (delay? %) (deref %) %)) ;; fix

(defn nav-path [p z-fn [tree id-fn children-fn]]
  (let [children-fn (comp handle-opt-delay children-fn)
        z-nav       (zn/path->z-nav tree id-fn children-fn p)
        p-next      (zn/z-nav->path (cond-> z-nav (seq p) z-fn)
                                    id-fn)]
    [z-nav p-next]))

(defn init-ctrl-bindings! [navtree-token k-fn-pairs]
  (doseq [[k f] k-fn-pairs]
    (kb/bind! k (str navtree-token k) f)))

(defn nav-to! [sp-ratom z-nav-args on-selection-path z-fn]
  (let [[_ sel-path-next] (nav-path @sp-ratom z-fn z-nav-args)]
    (tb/info "next selection-path: " sel-path-next)
    (on-selection-path sel-path-next)))

(defn init-nav-bindings!
  [sp-ratom z-nav-args on-selection-path navtree-token k-navfn-pairs]
  (doseq [[k z-fn] k-navfn-pairs]
    (kb/bind! k (str navtree-token k)
      (fn [& _] (nav-to! sp-ratom z-nav-args on-selection-path z-fn)))))

(defn unbind! [navtree-token kk]
  (doseq [k kk] (kb/unbind! k (str navtree-token k))))

(defn jump-to-content!
  [sp-ratom z-nav-args on-selection-path content-fn]
  (loop []
    (let [[_ id-fn _] z-nav-args
          sp-curr @sp-ratom
          [z-nav sp-next] (nav-path sp-curr z/next z-nav-args)
          content         (-> z-nav z/node id-fn content-fn handle-opt-delay)]
      (when-not content
        (if (= sp-curr sp-next)
          (tb/warn "next returns same selection path:" sp-next)
          (do (tb/info "jumping from:" sp-curr "to" sp-next)
              (on-selection-path sp-next)
              (recur)))))))

(defn -navtree
  [{:as params
    :keys [tree id-fn children-fn
           selection-path selection-path-ratom selection-path-init
           on-selection-path
           keynav? jump-to-content? expand-nav?
           whitelist whitelist?
           content-fn content-loading-indicator content-placeholder content-ratom]
    :or {expand-nav? true whitelist? true}}]
  (r/with-let
    [row-mode? (r/atom (not expand-nav?))
     ;; selection-path (root-headed)
     sp-ratom  (or selection-path-ratom
                   (r/atom selection-path-init))
     sp-update #(reset! sp-ratom %)

     c-ratom   (or content-ratom
                   (r/atom (or content-placeholder "")))

     ;; whitelist
     wl-ratom     (when whitelist? (r/atom #{}))
     wl-update-sp (fn [p] (reset! wl-ratom (set p)))
     _ (wl-update-sp @sp-ratom)

     ;; handler
     on-selection-path
     (or on-selection-path
         #(do (reset! c-ratom content-loading-indicator)
              (sp-update %)
              (when whitelist?
                (wl-update-sp %))))

     z-nav-args [tree id-fn children-fn]
     k-navfn-pairs
     (partition 2
       ["j" z/next "k" z/prev "h" z/up "shift-j" z/right "shift-k" z/left])
     k-fn-pairs
     (partition 2
       ["u" #(swap! row-mode? not)])
     navtree-token (name (gensym))
     _
     (when keynav?
       (init-nav-bindings! sp-ratom z-nav-args on-selection-path navtree-token k-navfn-pairs)
       (init-ctrl-bindings! navtree-token k-fn-pairs))
     _
     (when jump-to-content?
       (jump-to-content! sp-ratom z-nav-args on-selection-path content-fn))]

    (let [sp (or selection-path @sp-ratom)
          wl (or whitelist
                 (and whitelist?
                      (or (and selection-path (set selection-path))
                          @wl-ratom)))]
      [:div.zc-navtree
        {:class (when @row-mode? "row-mode")}
        (let [f (partial nav-to! sp-ratom z-nav-args on-selection-path)]
          [:div.controls
            [:span {:on-click #(swap! row-mode? not)} (if @row-mode? "expand" "collapse")]
            [:span.zmdi.zmdi-chevron-left  {:on-click (partial f z/prev)}]
            [:span.zmdi.zmdi-chevron-right {:on-click (partial f z/next)}]])
        [:ul.tree
          ^{:key (hash (id-fn tree))}
          [-navtree-node
            (-> params
                (assoc  :content-ratom c-ratom
                        :on-selection-path on-selection-path)
                (cond-> whitelist? (assoc :whitelist wl)
                        true       (assoc :selection-path sp))
                (dissoc :tree))
            (list)
            tree]]

        (when (and content-fn
                   (not content-ratom))
          @c-ratom)])
    (finally
      (when keynav? (unbind! navtree-token
                             (map first (concat k-fn-pairs k-navfn-pairs))))
      (reset! c-ratom nil))))

(defn navtree [& {:as params}] [-navtree params])
