(ns hara.image.awt.io
  (:require [clojure.java.io :as io])
  (:import (javax.imageio.spi IIORegistry
                              ImageInputStreamSpi
                              ImageReaderSpi
                              ImageTranscoderSpi
                              ImageWriterSpi)
           (javax.imageio ImageIO)
           (java.awt.image BufferedImage))
  (:refer-clojure :exclude [read]))

(def lookup
  {:reader ImageReaderSpi
   :writer ImageWriterSpi
   :transcoder ImageTranscoderSpi})

(defn providers
  "list providers for ImageIO
 
   (providers :reader)
   
   (providers :writer)"
  {:added "2.8"}
  ([] (providers :reader))
  ([type]
   (-> (IIORegistry/getDefaultInstance)
       (.getServiceProviders (lookup type) true)
       (iterator-seq))))

(defn supported-formats
  "list supported formats for ImageIO
 
   (supported-formats :reader)
   => (contains [\"BMP\" \"GIF\" \"JPEG\" \"JPG\" \"PNG\" \"WBMP\"] :in-any-order)"
  {:added "2.8"}
  ([] (supported-formats :reader))
  ([type]
   (->> (providers type)
        (mapcat #(.getFormatNames %))
        (map #(.toUpperCase %))
        set sort vec)))

(defn awt->int-argb
  "converts an indexed or custom image into and int-argb image
 
   (-> (ImageIO/read (java.io.File. \"resources/data/circle-10.gif\"))
       awt->int-argb)
   ;;#image.awt[10 10]{:model :int-argb}
   => java.awt.image.BufferedImage
   "
  {:added "2.8"}
  [image]
  (let [w (.getWidth image)
        h (.getHeight image)
        nimage (BufferedImage. w h BufferedImage/TYPE_INT_ARGB)]
    (doall (for [i (range h)
                 j (range w)]
             (.setRGB nimage j i (.getRGB image j i))))
    nimage))

(defn read
  "loads a BufferedImage from input
 
   (read \"resources/data/circle-100.png\")
   ;; #image.awt[100 100]{:model :4-byte-abgr}
   => java.awt.image.BufferedImage
   "
  {:added "2.8"}
  [input]
  (let [image (ImageIO/read (io/input-stream input))]
    (if (#{BufferedImage/TYPE_CUSTOM
           BufferedImage/TYPE_BYTE_INDEXED} (.getType image))
      (awt->int-argb image)
      image)))

(defn write
  "saves a BufferedImage to file or stream
 
   (-> (read \"resources/data/circle-100.png\")
       (write \"JPG\" \"resources/data/circle-100.jpg\"))
   => true"
  {:added "2.8"}
  [img output opts]
  (ImageIO/write img (:format opts) (io/output-stream output)))

