(ns hara.object.access
  (:require [hara.object
             [read :as read]
             [write :as write]])
  (:refer-clojure :exclude [get set]))

(defn get-with-keyword
  "access the fields of an object with keyword
 
   (get-with-keyword {:a 1} :a)
   => 1
 
   (get-with-keyword (test.Cat. \"spike\")
                     :species)
   => \"cat\""
  {:added "2.3"}
  [obj k]
  (if (instance? java.util.Map obj)
    (clojure.core/get obj k)
    (if-let [getter (-> obj type read/meta-read :methods k)]
      (getter obj))))

(defn get-with-array
  "access the fields of an object with an array of keywords
 
   (get-with-array {:a 1} [:a])
   => {:a 1}
 
   (get-with-array (test.Cat. \"spike\")
                     [:name :species])
   => {:name \"spike\", :species \"cat\"}"
  {:added "2.3"}
  [obj arr]
  (if (instance? java.util.Map obj)
    (select-keys obj arr)
    (let [getters (-> obj type read/meta-read :methods (select-keys arr))]
      (reduce-kv (fn [i k v]
                   (assoc i k (v obj)))
                 {}
                 getters))))

(defn get
  "accessor with either keyword or array lookup
 
   (access/get (test.Cat. \"spike\") :name)
   => \"spike\""
  {:added "2.3"}
  [obj k]
  (cond (keyword? k)
        (get-with-keyword obj k)
        
        (sequential? k)
        (get-with-array obj k)))

(defn set-with-keyword
  "sets the fields of an object with keyword
 
   (-> (doto (test.Cat. \"spike\")
         (set-with-keyword :name \"fluffy\"))
       (access/get :name))
   => \"fluffy\""
  {:added "2.3"}
  [obj k v]
  (if-let [setter (-> obj type write/meta-write :methods k)]
    ((:fn setter) obj v)
    (throw (Exception. (str "key " k " does not exist for " (.getName (type obj)))))))

(defn set
  "sets the fields of an object with a map
   
   (-> (doto (test.Cat. \"spike\")
         (access/set {:name \"fluffy\"}))
       (access/get :name))
   => \"fluffy\""
  {:added "2.3"}
  ([obj m]
   (reduce-kv (fn [obj k v]
                (set-with-keyword obj k v)
                obj)
              obj
              m)
   obj)
  ([obj k v]
   (set-with-keyword obj k v)
   obj))

