(ns hara.object.write
  (:require [clojure.walk :as walk]
            [hara.protocol.object :as object]
            [hara.protocol.map :as map]
            [hara.data.map :as data]
            [hara.string.case :as case]
            [hara.reflect :as reflect]
            [hara.reflect.util :as reflect-util]))

(def default-write-template
  '(fn <method> [obj val]
     (or (. obj (<method> val))
         obj)))

(defn meta-write
  "accesses the write-attributes of an object
 
   (write/meta-write DogBuilder)
   => (contains {:class test.DogBuilder
                 :empty fn?,
                 :methods (contains
                           {:name
                            (contains {:type java.lang.String, :fn fn?})})})"
  {:added "2.3"}
  [^Class cls]
  (assoc (object/-meta-write cls) :class cls))

(declare from-data)

(defn write-reflect-fields
  "write fields of an object from reflection
   (-> (write/write-reflect-fields Dog)
       keys)
   => [:name :species]"
  {:added "2.3"}
  [cls]
  (->> (reflect/query-class cls [:field])
       (reduce (fn [out ele]
                 (let [k (-> ele :name case/spear-case keyword)
                       cls (.getType (get-in ele [:all :delegate]))]
                   (assoc out k {:type cls :fn ele})))
               {})))

(defn create-write-method
  "create a write method from the template
 
   (-> ((-> (write/create-write-method (reflect/query-class Cat [\"setName\" :#])
                                       \"set\"
                                       write/default-write-template
                                       )
            second
            :fn) (test.Cat. \"spike\") \"fluffy\")
      (.getName))
   => \"fluffy\""
  {:added "2.3"}
  [ele prefix template]
  [(-> (:name ele) (subs (count prefix)) case/spear-case keyword)
   {:type (-> ele :params second)
    :fn (eval (walk/postwalk-replace {'<method> (symbol (:name ele))}
                                     template))}])

(defn write-setters
  "write fields of an object through setter methods
   (write/write-setters Dog)
   => {}
 
   (keys (write/write-setters DogBuilder))
   => [:name]"
  {:added "2.3"}
  ([cls] (write-setters cls {}))
  ([cls {:keys [prefix template]
         :or {prefix "set"
              template default-write-template}}]
   (->> [:method :instance (re-pattern (str "^" prefix ".+")) 2]
        (reflect/query-class cls)
        (reduce (fn [out ele]
                  (conj out (create-write-method ele prefix template)))
                {}))))

(defn write-all-setters
  "write all setters of an object and base classes
   (write/write-all-setters Dog)
   => {}
 
   (keys (write/write-all-setters DogBuilder))
   => [:name]"
  {:added "2.3"}
  ([cls] (write-all-setters cls {}))
  ([cls {:keys [prefix template]
         :or {prefix "set"
              template default-write-template}}]
   (->> [:method :instance (re-pattern (str "^" prefix ".+")) 2]
        (reflect/query-hierarchy cls)
        (reduce (fn [out ele]
                  (conj out (create-write-method ele prefix template)))
                {}))))

(defn from-empty
  "creates the object from an empty object constructor
   (write/from-empty {:name \"chris\" :pet \"dog\"}
                     (fn [_] (java.util.Hashtable.))
                     {:name {:type String
                             :fn (fn [obj v]
                                   (.put obj \"hello\" (keyword v))
                                   obj)}
                      :pet  {:type String
                            :fn (fn [obj v]
                                   (.put obj \"pet\" (keyword v))
                                   obj)}})
   => {\"pet\" :dog, \"hello\" :chris}"
  {:added "2.3"}
  [m empty methods]
  (let [obj (empty m)]
    (reduce-kv (fn [obj k v]
                 (if-let [{:keys [type] func :fn} (get methods k)]
                   (func obj (from-data v type))
                   obj))
               obj
               m)))

(defn from-constructor
  "creates the object from a constructor
   (-> {:name \"spike\"}
       (write/from-constructor {:fn (fn [name] (Cat. name))
                                :params [:name]}
                               {}))
   ;;=> #test.Cat{:name \"spike\", :species \"cat\"}
   "
  {:added "2.3"}
  [m {:keys [params] :as construct} methods]
  (let [obj (apply (:fn construct) (map m params))]
    (reduce-kv (fn [obj k v]
                 (if-let [{:keys [type] func :fn} (get methods k)]
                   (func obj (from-data v type))
                   obj))
               obj
               (apply dissoc m params))))

(defn from-map
  "creates the object from a map
   (-> {:name \"chris\" :age 30 :pets [{:name \"slurp\" :species \"dog\"}
                                     {:name \"happy\" :species \"cat\"}]}
       (write/from-map Person)
       (read/to-data))
   => (contains-in
       {:name \"chris\",
        :age 30,
       :pets [{:name \"slurp\"}
               {:name \"happy\"}]})"
  {:added "2.3"}
  [m ^Class cls]
  (let [m (if-let [rels (get object/*transform* type)]
            (data/transform-in m rels)
            m)
        {:keys [construct empty methods from-map] :as mobj} (meta-write cls)]
    (cond from-map
          (from-map m)

          empty
          (from-empty m empty methods)

          construct
          (from-constructor m construct methods)
          
          :else
          (map/-from-map m cls))))

(defn from-data
  "creates the object from data
   (-> (write/from-data [\"hello\"] (Class/forName \"[Ljava.lang.String;\"))
       seq)
   => [\"hello\"]"
  {:added "2.3"}
  [arg ^Class cls]
  (let [^Class targ (type arg)]
    (cond
      ;; If there is a direct match
      (reflect-util/param-arg-match cls targ)
      arg

      ;; Special case for String/CharArray
      (and (string? arg) (= cls (Class/forName "[C")))
      (.toCharArray arg)
      
      ;; If there is a vector
      (and (vector? arg)
           (.isArray cls))
      (let [cls (.getComponentType cls)]
        (->> arg
             (map #(from-data % cls))
             (into-array cls)))

      :else
      (let [{:keys [from-clojure from-string from-vector] :as mobj} (meta-write cls)]
        (cond

          from-clojure (from-clojure arg)

          ;; If input is a string and there is a from-string method
          (and (string? arg) from-string)
          (from-string arg)

          ;; If input is a string and there is a from-string method
          (and (vector? arg) from-vector)
          (from-vector arg)

          ;; If the input is a map
          (map? arg)
          (from-map arg cls)

          :else
          (throw (Exception. (format "Problem converting %s to %s" arg cls))))))))
