(ns lucid.insight.analyser
  (:require [clojure.java.classpath :as classpath]
            [clojure.java.io :as io]
            [clojure.tools.analyzer.ast :as ast]
            [clojure.tools.analyzer.env :as env]
            [clojure.tools.analyzer.jvm :as jvm]
            [clojure.tools.analyzer.jvm.utils :as jvm-utils]
            [clojure.tools.reader :as reader]
            [clojure.tools.reader.reader-types :as reader-types]
            [hara.io.project :as project]))

(def ^:dynamic *current-project* nil)

(defn try-analyze [form env opts]
  (try (jvm/analyze form env opts)
       (catch Throwable t
         (println t)
         (println "WARNING: skipping form: " form))))

(defn analyze-ns
  "Analyzes a whole namespace. returns a vector of the ASTs for all the
   top-level ASTs of that file.
   Evaluates all the forms.
   Disables wrong-tag-handler, and fixes bug with opts shadowing,
   and doesn't eval."
  ([ns] (analyze-ns ns (jvm/empty-env)))
  ([ns env] (analyze-ns ns env {:passes-opts
                                (merge
                                 jvm/default-passes-opts
                                 {:validate/wrong-tag-handler
                                  (fn [_ ast]
                                    #_(println "Wrong tag: " (-> ast :name meta :tag)
                                               " in def: " (:name ast)))})}))
  ([ns env opts]
   (println "Analyzing ns" ns)
   (env/ensure
    (jvm/global-env)
    (let [res ^java.net.URL (jvm-utils/ns-url ns)]
      (assert res (str "Can't find " ns " in classpath"))
      (let [filename (str res)
            path     (.getPath res)]
        (when-not (get-in (env/deref-env) [::analyzed-clj path])
          (binding [*ns*   (the-ns ns)
                    *file* filename]
            (with-open [rdr (io/reader res)]
              (let [pbr (reader-types/indexing-push-back-reader
                         (java.io.PushbackReader. rdr) 1 filename)
                    eof (Object.)
                    read-opts {:eof eof :features #{:clj :t.a.jvm}}
                    read-opts (if (.endsWith filename "cljc")
                                (assoc read-opts :read-cond :allow)
                                read-opts)]
                (loop [ret []]
                  (let [form (reader/read read-opts pbr)]
                    (if (identical? form eof)
                      (remove nil? ret)
                      (recur
                       (conj ret (try-analyze form (assoc env :ns (ns-name *ns*)) opts)))))))))))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Private helpers: extracting definitions and references from forms

(defn class-name [^Class c] (.getName c))

(defn gen-interface-class [f]
  (when (= (first f) 'clojure.core/gen-interface)
    [(name (nth f 2))]))

(defn protocol-gen-interface-form [node]
  (when (= (:type node) :class) (-> node :raw-forms first)))

(defn class-defs
  [{:keys [op] :as node}]
  (case op
    (:deftype) [(class-name (or (get node :class-name)
                                (throw (Exception. (str "Node: " node " does not contain :class-name")))))]
    (:import) (-> node :raw-forms first next next first gen-interface-class) ;; possibly definterface
    (:const) (-> node protocol-gen-interface-form gen-interface-class) ;; possibly defprotocol
    nil))

(defn class-refs [{:keys [op] :as node}]
  (case op
    (:const) (when (= (:type node) :class)
               [(class-name (or (get node :val)
                                (throw (Exception. (str "Node: " node " does not contain :val")))))])
    nil))

(defn var->symbol [v]
  (let [{:keys [ns] :as m} (meta v)]
    (symbol (name (ns-name ns)) (name (:name m)))))

(defn defprotocol-vars [f ns]
  (when (= (first f) 'clojure.core/gen-interface)
    (for [[m] (nth f 4)]
      (symbol (name ns) (name m)))))

(defn var-defs [{:keys [op] :as node}]
  (case op
    (:def) [(var->symbol (or (get node :var)
                             (throw (Exception. (str "Node: " node " does not contain :var")))))]
    
    (:const) (-> node
                 protocol-gen-interface-form
                 (defprotocol-vars (or (get-in node [:env :ns])
                                       (throw (Exception. (str "Node: " node " does not contain [:env :ns]"))))))
    
    (:instance-call) (when (and (= 'addMethod (:method node)) ;; defmethod is part of multi var def.
                                (= clojure.lang.MultiFn (:class node)))
                       [(var->symbol (or (get-in node [:instance :var])
                                         (throw (Exception. (str "Node: " node " does not contain [:instance :var]")))))])
    nil))

(defn var-refs [{:keys [op] :as node}]
  (case op
    (:var) (do [(var->symbol (or (get node :var)
                                 (throw (Exception. (str "Node: " node " does not contain :var")))))])
    nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Public

(defn normalized-form
  "Convert a top-level analyzer AST node into a sniper/Form.
   May return nil for comment forms (which are often missing source info)."
  [ast-node]
  (let [nodes (ast/nodes ast-node)
        unique (fn [f] (sort-by str (distinct (mapcat f nodes))))]
    (if-not (get-in ast-node [:env :line])
      (do (assert (= (ffirst (:raw-forms ast-node)) 'comment)
                  (str "MISSING line" (:raw-forms ast-node) (:env ast-node)))
          nil)
      {:source-info (select-keys (:env ast-node) [:file :ns :line :column :end-column :end-line])
       :class-defs  (unique class-defs)
       :class-refs  (unique class-refs)
       :var-defs    (unique var-defs)
       :var-refs    (unique var-refs)
       :shadow? false})))

(def +analysis-cache-file+ ".insight-cache.edn")

(defonce +analysis-cache+
  (atom (try (read-string (slurp +analysis-cache-file+)) (catch Exception e {}))))

(defn cached-ns-forms
  [ns]
  (let [c (slurp (jvm-utils/ns-url ns))]
    (or (@+analysis-cache+ c)
        (let [res (vec (keep normalized-form (analyze-ns ns)))]
          (swap! +analysis-cache+ assoc c res)
          res))))

(defn ns-forms
  "Get a flat sequence of forms for all namespaces in nss."
  [& nss]
  (apply concat (pmap cached-ns-forms nss)))

(defn all-forms
  ([selectors]
   (all-forms selectors (project/project)))
  ([{:keys [include exclude]} project]
   (binding [*current-project* project]
     (let [nss (->> (keys (project/all-files (:source-paths project) {} project))
                    (remove (fn [ns]
                              (some (fn [re] (re-find re (name ns))) exclude))))
           nss (if (empty? include)
                 nss
                 (filter (fn [ns]
                           (some (fn [re] (re-find re (name ns))) include))
                         nss))]
       (println "Requiring" nss)
       (apply require nss)
       (let [res (apply ns-forms nss)]
         (future (spit +analysis-cache-file+ @+analysis-cache+))
         res)))))


#_(comment
  (ns-forms 'lucid.unit)
  (def project (project/project))
  (keys )
  
  (classpath-namespaces #"src/lucid/unit")
  )
#_(comment
  (def res (classpath-ns-forms #"src/lucid/unit"))
  (def res (classpath-ns-forms #"src/lucid"))
  (+ 1 1)
  
  )

#_(comment
  (let [c (slurp (jvm-utils/ns-url ns))]
    (or (@+analysis-cache+ c)
        (let [res (vec (keep normalized-form (doto (analyze-ns ns)
                                               (prn "ANALYSIS"))))]
          
          (swap! +analysis-cache+ assoc c res)
          res)))
  (the-ns (namespace 'lucid.insight))
  
  (def result (analyze-ns 'lucid.insight))
  
  
  (normalized-form (first result))
  
  
  (prn (count ))
  (analyze-ns )
  
  )
