(ns lucid.publish.parse.checks
  (:require [lucid.query :as query]
            [rewrite-clj.zip :as source]
            [rewrite-clj.node :as node]))

(def directives
  #{:article :file :reference :ns
    :appendix :chapter
    :section :subsection :subsubsection
    :image :paragraph :code
    :equation :citation
    :html :api})

(defn wrap-meta
  "helper function for navigating `^:meta` tags
 
   ((wrap-meta query/match) (zip/of-string \"^:hello ()\") list?)
   => true"
  {:added "1.2"}
  [f]
  (fn [zloc selector]
    (if (= :meta (source/tag zloc))
      (f (-> zloc source/down source/right) selector)
      (f zloc selector))))

(defn directive?
  "check if element is a directive
   
   (directive? (zip/of-string \"[[:chapter {:title \\\"hello\\\"}]]\"))
   => true"
  {:added "1.2"}
  ([zloc]
   ((wrap-meta query/match) zloc {:pattern [[#'keyword? #'map?]]}))
  ([zloc kw]
   ((wrap-meta query/match) zloc {:pattern [[kw #'map?]]})))

(defn attribute?
  "check if element is an attribute
 
   (attribute? (zip/of-string \"[[{:title \\\"hello\\\"}]]\"))
   => true"
  {:added "1.2"}
  [zloc]
  ((wrap-meta query/match) zloc {:pattern [[#'map?]]}))

(defn code-directive?
  "check if element is a code directive
 
   (code-directive? (zip/of-string \"[[:code {:lang \\\"python\\\"} \\\"1 + 1\\\"]]\"))
   => true"
  {:added "1.2"}
  [zloc]
  ((wrap-meta query/match) zloc {:pattern [[:code #'map? #'string?]]}))

(defn ns?
  "check if element is a `ns` form
 
   (ns? (zip/of-string \"(ns lucid.unit)\"))
   => true"
  {:added "1.2"}
  [zloc]
  ((wrap-meta query/match) zloc {:form 'ns}))

(defn fact?
  "check if element is a `fact` form
 
   (fact? (zip/of-string \"(fact 1 => 1)\"))
   => true"
  {:added "1.2"}
  [zloc]
  ((wrap-meta query/match) zloc {:form 'fact}))

(defn facts?
  "check if element is a `facts` form
 
   (facts? (zip/of-string \"(facts 1 => 1)\"))
   => true"
  {:added "1.2"}
  [zloc]
  ((wrap-meta query/match) zloc {:form 'facts}))

(defn comment?
  "check if element is a `comment` form
 
   (comment? (zip/of-string \"(comment 1 => 1)\"))
   => true"
  {:added "1.2"}
  [zloc]
  ((wrap-meta query/match) zloc {:form 'comment}))

(defn deftest?
  "check if element is a `deftest` form
 
   (deftest? (zip/of-string \"(deftest ...)\"))
   => true"
  {:added "1.2"}
  [zloc]
  ((wrap-meta query/match) zloc {:form 'deftest}))

(defn is?
  "check if element is an `is` form
 
   (is? (zip/of-string \"(is ...)\"))
   => true"
  {:added "1.2"}
  [zloc]
  ((wrap-meta query/match) zloc {:form 'is}))

(defn paragraph?
  "check if element is a paragraph (string)
 
   (paragraph? (zip/of-string \"\\\"hello world\\\"\"))
   => true"
  {:added "1.2"}
  [zloc]
  (string? (source/sexpr zloc)))

(defn whitespace?
  "check if element is whitespace
 
   (whitespace? (zip/next* (zip/of-string \"a b\")))
   => true"
  {:added "1.2"}
  [zloc]
  (node/whitespace-or-comment? (source/node zloc)))
