(ns lucid.publish.collect.reference
  (:require [lucid.core.code :as code]
            [hara.io.file :as fs]
            [hara.data.nested :as nested]))

(defn find-import-namespaces
  "finds forms with `(ns/import ...)`
 
   (find-import-namespaces (project/file-lookup (project/project))
                           'lucid.publish)
   => '(lucid.publish.theme)"
  {:added "1.2"}
  [lookup ns]
  (if-let [path (lookup ns)]
    (->> (fs/code path)
         (filter #(-> % first (= 'ns/import)))
         (mapcat #(->> % rest (take-nth 2))))))

(defn reference-namespaces
  "finds the referenced vars in the namespace
 
   (-> (reference-namespaces {}
                             (project/file-lookup (project/project))
                             '[lucid.core.asm])
       (get 'lucid.core.asm)
       keys
       sort)
  => '(dynamic-loader load-class path->classname to-bytes unload-class)"
  {:added "1.2"}
  [references lookup namespaces]
  (let [missing   (remove references namespaces)
        imported  (->> missing
                       (mapcat #(find-import-namespaces lookup %))
                       (remove references))
        sources   (concat missing imported)
        tests     (map #(symbol (str % "-test")) sources)]
    (reduce (fn [references ns]
              (if-let [file (lookup ns)]
                (->> (code/analyse-file file)
                     (nested/merge-nested references))
                references))
            references
            (concat sources tests))))

(defn collect-references
  "collects all `:reference` tags of within an article
   
   (let [project (project/project)
         project (assoc project :lookup (project/file-lookup project))
         elems   (parse/parse-file \"test/documentation/lucid_publish.clj\" project)
         bundle  {:articles {\"lucid-publish\" {:elements elems}}
                  :references {}
                  :project project}]
     (-> (collect-references bundle \"lucid-publish\")
         :references
        keys))
   => '(lucid.publish.theme)"
  {:added "1.2"}
  [{:keys [articles project] :as interim} name]
  (let [all    (->> (get-in articles [name :elements])
                    (filter #(-> % :type (= :reference))))
        namespaces (-> (map (comp symbol namespace symbol :refer) all))]
    (-> interim
        (update-in [:references]
                   (fnil (fn [references]
                           (reference-namespaces references
                                                 (:lookup project)
                                                 namespaces))
                         {})))))
