(ns spirit.core.graph
  (:require [spirit.protocol.igraph :as graph]
            [hara.component :as component])
  (:refer-clojure :exclude [update]))

(defn install-schema
  "installs the schema within the graph database
 
   (-> (graph {:type :datomic
               :uri \"datomic:mem://graph-test\"
               :options {:reset-db true}})
       (graph/install-schema {:user  {:email [{:required true}]
                                      :firstname [{}]
                                      :lastname  [{}]
                                      :hash      [{}]}}))
   ;; #datomic{:type :datomic,
   ;;          :uri \"datomic:mem://graph-test\",
   ;;          :schema #schema{...},
   ;;          :options {:install-schema true, :reset-db true},
   ;;          :connection #conn{1000 #inst \"2018-06-26T00:55:42.316-00:00\"}}
   "
  {:added "0.9"}
  [db schema]
  (graph/-install-schema db schema))

(defn purge
  "destroys all the data within a graph database
 
   (-> (graph db-config)
       (graph/insert [{:user {:email \"a@a.com\"}}
                      {:user {:email \"b@b.com\"}}])
       (graph/purge)
       (graph/select {:user '_}))
   => #{}"
  {:added "0.9"}
  ([db]
   (purge db {}))
  ([db opts]
   (graph/-purge db opts)))

(defn select
  "given a query, selects the relevent results
 
   (-> (graph db-config)
       (graph/insert [{:user {:email \"a@a.com\"}}
                      {:user {:email \"b@b.com\"}}])
       (graph/select {:user '_}))
   => #{{:user {:email \"a@a.com\"}}
        {:user {:email \"b@b.com\"}}}"
  {:added "0.9"}
  ([db selector]
   (select db selector {}))
  ([db selector opts]
   (graph/-select db selector opts)))

(defn insert
  "inserts into the graph database
 
   (-> (graph db-config)
       (graph/insert {:user {:email \"a@a.com\"}})
       (graph/select :user))
   => #{{:user {:email \"a@a.com\"}}}"
  {:added "0.9"}
  ([db data]
   (insert db data {}))
  ([db data opts]
   (graph/-insert db data opts)))

(defn delete
  "deletes an entry in the graph database
   
   (-> (graph db-config)
       (graph/insert [{:user {:email \"a@a.com\"}}
                      {:user {:email \"b@b.com\"}}])
       (graph/delete {:user {:email \"a@a.com\"}})
       (graph/select :user))
   => #{{:user {:email \"b@b.com\"}}}"
  {:added "0.9"}
  ([db selector]
   (delete db selector {}))
  ([db selector opts]
   (graph/-delete db selector opts)))

(defn update
  "updates an entry in the graph database
 
   (-> (graph db-config)
       (graph/insert [{:user {:email \"a@a.com\"}}
                      {:user {:email \"b@b.com\"}}])
       (graph/update {:user {:email \"a@a.com\"}}
                    {:user {:firstname \"Allan\"
                             :lastname \"Adams\"}})
       (graph/select :user))
   #{{:user {:email \"a@a.com\", :firstname \"Allan\", :lastname \"Adams\"}}
     {:user {:email \"b@b.com\"}}}"
  {:added "0.9"}
  ([db selector data]
   (update db selector data {}))
  ([db selector data opts]
   (graph/-update db selector data opts)))

(defn retract
  "retracts a property of an entry in the graph database
   
   (-> (graph db-config)
       (graph/insert [{:user {:email \"a@a.com\",
                              :firstname \"Allan\",
                              :lastname \"Adams\"}}
                      {:user {:email \"b@b.com\",
                              :firstname \"Billy\",
                              :lastname \"Bob\"}}])
       (graph/retract :user [:user/firstname])
      (graph/select :user))
   => #{{:user {:email \"a@a.com\", :lastname \"Adams\"}}
        {:user {:email \"b@b.com\", :lastname \"Bob\"}}}"
  {:added "0.9"}
  ([db selector keys]
   (retract db selector keys {}))
  ([db selector keys opts]
   (graph/-retract db selector keys opts)))

(defn create
  "creates an graph database instance for use with component
 
   (graph/create {:type :datomic
                  :uri \"datomic:mem://graph-test\"
                  :schema  {:user  {:email [{:required true}]
                                    :firstname [{}]
                                    :lastname  [{}]
                                    :hash      [{}]}}})
   ;; #datomic{:type :datomic,
   ;;          :uri \"datomic:mem://graph-test\",
   ;;          :schema {...}}
  "
  {:added "0.9"}
  [m]
  (graph/-create m))

(defn graph
  "creates an active graph database instance
   
   (graph {:type :datomic
           :uri \"datomic:mem://graph-test\"
           :schema  {:user  {:email [{:required true}]
                                    :firstname [{}]
                                    :lastname  [{}]
                                    :hash      [{}]}}
           :options {:install-schema true
                     :reset-db true}})
   ;; #datomic{:type :datomic,
   ;;          :uri \"datomic:mem://graph-test\",
   ;;          :schema #schema{...},
   ;;          :options {:install-schema true, :reset-db true},
   ;;          :connection #conn{1000 #inst \"2018-06-26T00:55:42.316-00:00\"}}
   "
  {:added "0.9"}
  [m]
  (-> (create m)
      (component/start)))


