(ns spirit.io.datomic.api.depack
  (:require [spirit.io.datomic.process.unpack :as unpack]
            [spirit.io.datomic.data :as data]
            [spirit.io.datomic.data.checks :as checks]
            [hara.common
             [checks :refer [long? hash-map?]]
             [error :refer [error]]]
            [hara.data.map :refer [assoc-in-if]]
            [hara.string.path :as path]))

(declare depack)

(defn depack-ref
  "special case for `:ref` type attributes
 
   (depack-ref 1
               {:ref {:ns :account}}
               (scaffold/account-db))
   => 1
 
   (depack-ref {:account/name \"A\"}
               {:ref {:ns :account}}
               (scaffold/account-db))
   => {:name \"A\"}"
  {:added "0.9"}
  [data attr datasource]
  (let [ns  (-> attr :ref :ns)]
    (cond (nil? data) (error "CANNOT be NIL:" attr)

          (hash-map? data)
          (unpack/strip-ns (depack data datasource) ns)

          (long? data) data

          (checks/db-id? data) (data/iid-seed data)

          :else
          (error "RETURN_REF: Cannot process data: " data))))

(defn wrap-depack-sets
  "wrapper for multiple values in a set
 
   ((wrap-depack-sets (fn [x _ _] (:item/name x)))
    #{{:db {:id 1}
       :item/name \"C\"}
      {:db {:id 2}
       :item/name \"D\"}}
    nil
   nil)
   => #{\"C\" \"D\"}"
  {:added "0.9"}
  [f]
  (fn [data attr datasource]
    (cond (set? data)
          (->> data
               (map #(f % attr datasource))
               (filter identity)
               (set))
          :else (f data attr datasource))))

(defn depack-loop
  "main loop for depacking returned objects
   
   (depack-loop {:db {:id 1}
                 :account/user \"A\",
                 :account/orders #{{:db {:id 2}
                                    :order/number 2,
                                    :order/items #{{:db {:id 3}
                                                    :item/name \"C\"}
                                                   {:db {:id 4}
                                                    :item/name \"D\"}}}
                                  {:db {:id 5}
                                    :order/number 1,
                                    :order/items #{{:db {:id 6}
                                                    :item/name \"A\"}
                                                   {:db {:id 7}
                                                    :item/name \"B\"}}}}}
                (scaffold/order-db))
   => {:db {:id 1}
       :account {:user \"A\"
                 :orders #{{:+ {:db {:id 2}}
                            :number 2,
                            :items #{{:name \"C\", :+ {:db {:id 3}}}
                                     {:name \"D\", :+ {:db {:id 4}}}
                                     }}
                           {:+ {:db {:id 5}}
                            :number 1,
                            :items #{{:name \"A\", :+ {:db {:id 6}}}
                                     {:name \"B\", :+ {:db {:id 7}}}}}}}}"
  {:added "0.9"}
  [data datasource]
  (reduce-kv
   (fn [out k v]
     (if-let [[attr] (-> datasource :schema :flat (get k))]
       (cond (= :ref (-> attr :type))
             (assoc-in-if out (path/split k)
                          ((wrap-depack-sets depack-ref) (get data k) attr datasource))


             (= :enum (-> attr :type))
             (assoc-in-if out (path/split k)
                          (if-let [ens (-> attr :enum :ns)]
                            (-> data (get k) path/split last)
                            (get data k)))

             :else
             (assoc-in-if out (path/split k) (get data k)))
       (assoc out k v)))
   {} data))

(defn depack
  "top level depack converting data into nested form
 
   (depack [{:account/user \"A\"}
            {:account/user \"B\"}]
           (scaffold/order-db))
   => [{:account {:user \"A\"}}
       {:account {:user \"B\"}}]"
  {:added "0.9"}
  [data datasource]
  (cond (vector? data)
        (mapv #(depack % datasource) data)

        (hash-map? data)
        (depack-loop data datasource)

        :else data))
