(ns spirit.io.datomic.api.model
  (:require [hara.common
             [error :refer [error]]
             [checks :refer [hash-map?]]]
            [hara.data.nested :refer [merge-nested]]
            [hara.string.path :as path]))

(declare model-input)

(defn model-input-branch-directive
  "switch at branch, treating :ref types as unchecked by default
 
   (model-input-branch-directive  [{:type :long}] :checked)
   => :checked
   
   (model-input-branch-directive [{:type :ref}] :checked)
   => :unchecked"
  {:added "0.9"}
  [[attr] directive]
  (if (= :ref (:type attr))
    :unchecked
    directive))

(defn model-input-branch-schema
  "creates a map of models that are checked
 
   (model-input-branch-schema {:orders [{:type :ref}]
                               :user   [{:type :string}]
                               :tags   [{:type :string}]}
                              :checked)
   => {:orders :unchecked, :user :checked, :tags :checked}
   
   (model-input-branch-schema {:account [{:type :ref}],
                               :items   [{:type :ref}],
                               :number  [{:type :long}]} :checked)
   => {:account :unchecked, :items :unchecked, :number :checked}"
  {:added "0.9"}
  ([subsch directive]
   (model-input-branch-schema subsch directive {}))
  ([subsch directive output]
   (if-let [[k v] (first subsch)]
       (cond (hash-map? v)
             (recur (next subsch) directive
                    (assoc output k (model-input-branch-schema v directive)))
             (vector? v)
             (recur (next subsch) directive
                    (assoc output k (model-input-branch-directive v directive)))
             :else
             (error "MODEL_INPUT_BRANCH_SCHEMA: " v " for key "
                    k " of " subsch " should be a hashmap (branch) or vector (attr)"))
       output)))

(defn model-input-branch
  "returns the checked items for a model
 
   (model-input-branch  {:orders :checked}
                        :checked
                        {:orders [{:type :ref, :ref {:ns :order}}],
                         :user [{:type :string}], :tags [{:type :string}]}
                        (-> (schema/schema examples/account-orders-items-image)
                            :tree))
  => {:orders {:account :unchecked
                :items :unchecked
                :number :checked}
       :user :checked, :tags :checked}"
  {:added "0.9"}
  [v dft subsch tsch]
  (cond (hash-map? v)
        (merge-nested
         (model-input-branch-schema subsch dft)
         (model-input v dft subsch tsch {}))
        
        (or (= v :checked) (= v :unchecked))
        (model-input-branch-schema subsch v)

        :else
        (error "MODEL_INPUT_BRANCH: Branches only take :checked and :unchecked as directives")))

(defn model-input-attr-ref
  "returns the return model for a ref
 
   (model-input-attr-ref :checked
                         :checked
                         [{:type :ref,
                           :ref {:ns :order}}]
                         (-> (schema/schema examples/account-orders-items-image)
                              :tree))
   => {:account :unchecked
       :items :unchecked
      :number :checked}
   "
  {:added "0.9"}
  [v dft [attr] tsch]
  (cond (or (= v :unchecked) (= v :id) (= v :yield))
        v

        (= v :checked)
        (let [nsk (-> attr :ref :ns)
              subsch (if nsk (get tsch nsk) tsch)]
          (model-input-branch-schema subsch :checked))

        (hash-map? v)
        (let [nsk (-> attr :ref :ns)
              subsch (if nsk (get tsch nsk) tsch)]
          (model-input-branch v dft subsch tsch))

        :else
        (error "MODEL_INPUT_ATTR_REF: Values only take :checked, :unchecked, :id and :yield as directives")))

(defn model-input-attr
  "switches between :ref and other types on inspection of [attr]
 
   (model-input-attr :checked :checked
                     [{:type :string, :cardinality :one, :ident :item/name}]
                     (-> (schema/schema examples/account-orders-items-image)
                         :tree))
   => :checked
   "
  {:added "0.9"}
  [v dft [attr] tsch]
  (if (= :ref (:type attr))
    (model-input-attr-ref v dft [attr] tsch)
    (if (or (= v :checked) (= v :unchecked))
      v
      (error "MODEL_INPUT_ATTR: Values only take :checked and :unchecked as directives"))))

(defn model-input
  "returns the expanded model input for a schema
 
   (model-input {:account {:orders {:items {:name :checked}}}}
                :checked
                (-> (schema/schema examples/account-orders-items-image)
                    :tree))
   => {:account {:user :checked
                 :tags :checked
                :orders {:account :unchecked
                          :number :checked
                          :items {:name :checked
                                  :order :unchecked
                                  :images :unchecked}}}}"
  {:added "0.9"}
  ([tmodel dft tsch]
   (model-input tmodel dft tsch tsch {}))
  ([tmodel dft psch tsch output]
   (if-let [[k v] (first tmodel)]
     (let [subsch (get psch k)]
       (cond (hash-map? subsch)
             (recur (next tmodel) dft psch tsch
                    (assoc output k (model-input-branch v dft subsch tsch)))

             (vector? subsch)
             (recur (next tmodel) dft psch tsch
                    (assoc output k (model-input-attr v dft subsch tsch)))

             :else
             (error "MODEL_INPUT: " v " for key " k " should be a hashmap (branch) or vector (attr)")))
     output)))

(declare model-unpack)

(defn model-unpack-branch
  "converts a shorthand version into the datomic model
 
   (model-unpack-branch :items
                        {:order :unchecked, :images :unchecked, :name :checked}
                        (-> (schema/schema examples/account-orders-items-image)
                            :tree
                            :order)
                        [:order]
                        (-> (schema/schema examples/account-orders-items-image)
                            :tree)
                       {})
   => {:order/items {:item/order :unchecked, :item/images :unchecked, :item/name :checked}}
   "
  {:added "0.9"}
  [k v sch nsv tsch output]
  (if-let [subsch (get sch k)]
   (cond (hash-map? subsch)
         (model-unpack v subsch (conj nsv k) tsch output)

         (vector? subsch)
         (let [[attr] subsch]
           (if (= :ref (:type attr))
             (let [nnsv (path/split (-> attr :ref :ns))]
               (assoc output (path/join (conj nsv k))
                      (model-unpack v (get-in tsch nnsv) nnsv tsch {})))
             (error "MODEL_RETURN_BRANCH: Attribute can only be a ref" )))

         :else
         (error "MODEL_RETURN_BRANCH: subsch - " v " - has to be a hashmap or vector"))

   (error "MODEL_RETURN_BRANCH: Schema does not exist for " (conj nsv k))))

(defn model-unpack-attr
  "unpacks an attribute given a schema
 
   (model-unpack-attr :number
                      :checked
                      (-> (schema/schema examples/account-orders-items-image)
                          :tree
                          :order)
                      [:order]
                     {})
   => {:order/number :checked}"
  {:added "0.9"}
  [k v sch nsv output]
  (if-let [subsch (get sch k)]
    (let [[attr] subsch]
      (cond (and (= :ref (:type attr))
                 (= :checked v))
            (error "MODEL_RETURN_ATTR: Allow cannot be a directive for a ref")
            
            :else
            (assoc output (path/join (conj nsv k)) v)))
    (error "MODEL_RETURN_ATTR: Schema does not exist for " (conj nsv k) sch)))

(defn model-unpack
  "unpacks a model into a style for datomic
 
   (model-unpack {:account {:user :checked
                            :tags :checked
                            :orders {:account :unchecked
                                     :number :checked
                                     :items {:name :checked
                                             :order :unchecked
                                            :images :unchecked}}}}
                 (-> (schema/schema examples/account-orders-items-image)
                     :tree))
   => {:account/orders {:order/account :unchecked,
                        :order/items {:item/order :unchecked,
                                      :item/images :unchecked,
                                      :item/name :checked},
                        :order/number :checked},
       :account/user :checked
       :account/tags :checked}"
  {:added "0.9"}
  ([tmodel tsch]
   (model-unpack tmodel tsch [] tsch {}))
  ([tmodel sch nsv tsch output]
   (if-let [[k v] (first tmodel)]
     (recur (next tmodel) sch nsv tsch
            (cond (= :+ k)
                  (merge (model-unpack v tsch) output)

                  (hash-map? v)
                  (model-unpack-branch k v sch nsv tsch output)

                  (keyword? v)
                  (model-unpack-attr k v sch nsv output)

                  :else
                  (error "MODEL_RETURN: v - " v " - has to be a hashmap, vector or keyword")))
     output)))
