(ns spirit.io.datomic.api.nested
  (:require [hara.string.path :as path]
            [hara.common.checks :refer [long?]]
            [clojure.walk :as walk]
            [spirit.io.datomic.api
             [prepare :as prepare]
             [retract :as retract]
             [select :as select]
             [transaction :as transaction]]
            [hara.event :refer [raise]]))

(defn search-path-analysis
  "creates a search term for the given path
 
   (search-path-analysis [[:store/inventories {:book/name \"Tom Sawyer\"}]]
                         (-> (schema/schema examples/bookstore)
                             :tree))
   => [{:val :inventories
        :rval :store
        :ns :inventory
        :term {:book/name \"Tom Sawyer\"}}]
 
   (search-path-analysis [[:address/stores   {:name \"Koala Books\"}]
                          [:inventories      {:book/name \"Tom Sawyer\"}]]
                         (-> (schema/schema examples/bookstore)
                             :tree))
   => [{:val :stores :rval :address :ns :store :term {:name \"Koala Books\"}}
       {:val :inventories :rval :store :ns :inventory :term {:book/name \"Tom Sawyer\"}}]"
  {:added "0.9"}
  ([spath tsch]
   (search-path-analysis spath tsch nil []))
  ([spath tsch last-res all-res]
     (if-let [[k v] (first spath)]
       (let [ks (path/split k)
             ks (if-let [nss (:ns last-res)]
                  (cons nss ks) ks)
             [attr] (get-in tsch ks)
             res  (-> (:ref attr)
                      (select-keys  [:val :rval :ns])
                      (assoc :term v))]
         (recur (next spath) tsch res (conj all-res res)))
       all-res)))

(defn build-search-term
  "creates the path for search from search term
 
   (build-search-term 'STORE
                      [{:val :inventories
                        :rval :store
                        :ns :inventory
                        :term {:book/name \"Tom Sawyer\"}}])
   => {:inventory {:store 'STORE :book/name \"Tom Sawyer\"}}
 
   (build-search-term 'ADDRESS
                      [{:val :stores
                        :rval :address
                        :ns :store
                        :term {:name \"Koala Books\"}}
                       {:val :inventories
                        :rval :store
                       :ns :inventory
                        :term {:book/name \"Tom Sawyer\"}}])
   => {:inventory {:store {:address 'ADDRESS :name \"Koala Books\"}
                   :book/name \"Tom Sawyer\"}}"
  {:added "0.9"}
  [vterm all-res]
  (let [[res & next-res] all-res]
    (if-not res
      vterm
      (let [_    (if-not (:ns res)
                   (raise [:cannot-perform-reverse-lookup {:value res}]))
            mterm (:term res)
            mterm (cond (long? mterm)
                        (if (empty? next-res) {:db/id mterm} {:+/db/id mterm})

                        (= '_ mterm) nil

                        :else mterm
                   )
            nterm (merge (assoc {} (:rval res) vterm) mterm)
            nterm (if (empty? next-res)
                    {(:ns res) nterm}
                    nterm)]
        (recur nterm next-res)))))

(defn build-search-term-form
  "creates a form for search used in evalutation
 
   (build-search-term-form
    [{:val :inventories,
      :rval :store,
      :ns :inventory,
      :term {:book/name \"Tom Sawyer\"}}])
   => (contains-in (fn [symbol?]
                    {:inventory {:store symbol?,
                                  :book/name \"Tom Sawyer\"}}))"
  {:added "0.9"}
  [all-res]
  (let [id (gensym)
        data (build-search-term id all-res)
        data (walk/postwalk (fn [x] (if (or (= x '_) (list? x))
                                   (list 'quote x)
                                   x))
                       data)]
    (list 'fn [id] data)))

(defn build-search-term-fn
  "function to construct the search term function"
  {:added "0.9"}
  [all-res]
  (eval (build-search-term-form all-res)))

(defn update-in!
  "updates the last accessed element using selector
 
   (-> (scaffold/bookstore-db)
       (update-in! {:store/name \"Koala Books\"}
                   [:store/inventories {:book/name \"Tom Sawyer\"}]
                   {:count 4}
                   {}))
   => (contains-in #{{:inventory {:count 4, :cover :hard}, :db {:id number?}}})"
  {:added "0.9"}
  [datasource data path update opts]
  (assert (even? (count path)) "The path must have a even number of items.")
  (let [datasource (prepare/prepare datasource opts data)
        ids (select/select datasource data {:options {:raw false}
                                     :return :ids})
        spath (partition 2 path)
        svec  (search-path-analysis spath (-> datasource :schema :tree))
        ndata-fn (build-search-term-fn svec)
        last-ns (:ns (last svec))
        update (if last-ns {last-ns update} update)
        output  (mapcat
                 #(transaction/update! (dissoc datasource :pipeline) (ndata-fn %) update
                                       {:options {:raw true
                                                  :ban-body-ids false
                                                  :ban-ids false
                                                  :ban-top-id false}}) ids)
        sids (map :db/id output)
        transact (-> datasource
                     (assoc :transact :datomic)
                     (assoc-in [:process :emitted] output)
                     (transaction/transact-fn))]
    (if (or (-> datasource :transact (= :datomic))
            (-> datasource :options :raw))
      transact
      (select/select (assoc datasource :db (:db-after transact))
                     (set sids)
                     (merge opts {:options {:debug false
                                            :ban-ids false
                                            :ban-top-id false
                                            :ids true}})))))

(defn delete-in!
  "deletes the last accessed element using selector
 
   (-> (scaffold/bookstore-db)
       (delete-in! {:store/name \"Koala Books\"}
                   [:store/inventories {:book/name \"Tom Sawyer\"}]
                   {}))
   => (contains-in #{{:inventory {:count 5, :cover :hard}, :db {:id number?}}})"
  {:added "0.9"}
  [datasource data path opts]
  (assert (even? (count path)) "The path must have a even number of items.")
  (let [datasource (prepare/prepare datasource opts data)
        ids (select/select datasource data {:options {:raw false}
                                     :return :ids} )
        spath (partition 2 path)
        svec  (search-path-analysis spath (-> datasource :schema :tree))
        ndata-fn (build-search-term-fn svec)
        output  (mapcat
                 #(transaction/delete! (dissoc datasource :pipeline) (ndata-fn %)
                                       {:options {:raw true
                                                  :ban-body-ids false
                                                  :ban-ids false
                                                  :ban-top-id false}}) ids)
        sids (map second output)
        transact (-> datasource
                     (assoc :transact :datomic)
                     (assoc-in [:process :emitted] output)
                     (transaction/transact-fn))]
    (if (or (-> datasource :transact (= :datomic))
            (-> datasource :options :raw))
      transact
      (select/select (assoc datasource :db (:db-before transact))
                     (set sids)
                     (merge opts {:options {:debug false
                                            :ban-ids false
                                            :ban-top-id false
                                            :ids true}})))))


(defn add-ns-entry
  "adds the namespace to the keyword or path
 
   (add-ns-entry :inventory :cover)
   => :inventory/cover"
  {:added "0.9"}
  [ns entry]
  (cond (vector? entry)
        [(path/join [ns (first entry)]) (second entry)]
        (keyword? entry) (path/join [ns entry])))

(defn retract-in!
  "retracts an attribute of last accessed element using selector
 
   (-> (scaffold/bookstore-db)
       (retract-in! {:store/name \"Koala Books\"}
                    [:store/inventories {:book/name \"Tom Sawyer\"}]
                    [:cover]
                    {}))
   => (contains-in #{{:inventory {:count 5}, :db {:id number?}}})"
  {:added "0.9"}
  [datasource data path retracts opts]
  (assert (even? (count path)) "The path must have a even number of items.")
  (let [datasource (prepare/prepare datasource opts data)
        ids (select/select datasource data {:options {:first false
                                                      :raw false}
                                            :return :ids})
        spath (partition 2 path)
        svec  (search-path-analysis spath (-> datasource :schema :tree))
        ndata-fn (build-search-term-fn svec)
        last-ns (:ns (last svec))
        nretracts (set (map #(add-ns-entry last-ns %) retracts))

        output  (mapcat
                 (fn [id] (retract/retract! (dissoc datasource :pipeline)
                                           (ndata-fn id)
                                           nretracts
                                           {:options {:raw true
                                                      :ban-body-ids false
                                                      :ban-ids false
                                                      :ban-top-id false}})) ids)
        sids (map second output)
        transact (-> datasource
                     (assoc :transact :datomic)
                     (assoc-in [:process :emitted] output)
                     (transaction/transact-fn))]
    (if (or (-> datasource :options :raw)
            (-> datasource :transact (= :datomic)))
      transact
      (select/select (assoc datasource :db (:db-after transact))
                     (set sids)
                     (merge opts {:options {:debug false
                                            :ban-ids false
                                            :ban-top-id false
                                            :ids true}})))))
