(ns spirit.io.datomic.api.prepare
  (:require [hara.common.checks :refer [boolean?]]
            [hara.data
             [map :refer [assoc-nil]]
             [nested :refer [merge-nested]]]
            [datomic.api :as datomic]
            [spirit.io.datomic.api.model :as model]))

(defn prepare-db
  "prepares the `:db` value of the datasource
 
   (-> (scaffold/account-db)
       (prepare-db)
       :db)
   ;;=> #db{1000 #inst \"2018-06-28T01:00:41.946-00:00\"}
   => datomic.db.Db"
  {:added "0.9"}
  [datasource]
  (let [db (or (:db datasource) (datomic/db (:connection datasource)))
        db (if-let [t (:at datasource)] (datomic/as-of db t) db)]
    (assoc datasource :db db)))

(defn model-access
  "creates a pipeline model given `:access`
   
   (model-access nil
                 {:account {:age :checked}}
                 :unchecked
                 (-> (schema/schema examples/account-name-age-sex) :tree))
   => {:allow {:account {:name :unchecked
                         :age :checked
                         :sex :unchecked}}
       :pull {:account/name :unchecked
             :account/age :checked
              :account/sex :unchecked}}"
  {:added "0.9"}
  [model access dft tsch]
  (cond (and (:allow model) (:pull model))
        model

        (and (not (:allow model)) (:pull model))
        (assoc model :allow (model/model-input access dft tsch))

        :else
        (let [imodel (model/model-input access dft tsch)
              rmodel (model/model-unpack imodel tsch)]
          (assoc-nil model :allow imodel :pull rmodel))))

(defn model-pull
  "creates a pipeline model given `:pull`
 
   (model-pull nil
               {:account {:age :unchecked}}
               :checked
               (-> (schema/schema examples/account-name-age-sex) :tree))
   => {:pull {:account/name :checked
              :account/age :unchecked
             :account/sex :checked}}"
  {:added "0.9"}
  [model pull dft tsch]
  (assoc model :pull
         (-> pull
             (model/model-input dft tsch)
             (model/model-unpack tsch))))

(defn prepare-model
  "prepares both the `:access` and `:pull` models
 
   (-> (scaffold/account-db)
       (assoc :access {:account {:age :unchecked}})
       (prepare-model)
       :pipeline)
   => {:allow {:account {:name :checked
                         :age :unchecked
                        :sex :checked}}
       :pull {:account/name :checked
              :account/age :unchecked
              :account/sex :checked}}"
  {:added "0.9"}
  [datasource]
  (let [op    (:op datasource)
        dft   (if (-> datasource :options :blank)
                :unchecked 
                :checked)
        model  (or (:pipeline datasource) (if op (-> datasource :profile op)))
        model  (if-let [access (:access datasource)]
                 (model-access model access dft (-> datasource :schema :tree))
                 model)
        model  (if-let [pull (:pull datasource)]
                 (model-pull model pull dft (-> datasource :schema :tree))
                 model)]
    (assoc datasource :pipeline model)))

(defn prepare
  "Prepares `:db`, `:model`, `:input` and additional options
 
   (-> (prepare (scaffold/account-db)
                {:pull {:account {:age :unchecked}}}
                :account/name)
       (select-keys [:db :pipeline :process :tempids]))
   => (contains {:db datomic.db.Db
                 :pipeline {:pull {:account/name :checked
                                  :account/age :unchecked
                                   :account/sex :checked}}
                 :process {:input :account/name}
                 :tempids clojure.lang.Atom})"
  {:added "0.9"}
  [datasource opts input]
  (-> datasource
      (merge-nested opts)
      (prepare-db)
      (prepare-model)
      (assoc-in [:process :input] input)
      (assoc-in [:tempids] (atom #{}))))
