(ns spirit.io.datomic.api.select
  (:require [hara.common
             [checks :refer [hash-map? long?]]
             [error :refer [error]]]
            [hara.data.map :refer [assoc-nil]]
            [spirit.io.datomic.types :as types]
            [spirit.io.datomic.api
             [prepare :as prepare]]
            [spirit.io.datomic.process
             [pipeline :as pipeline]
             [pack :as pack]
             [emit :as emit]
             [unpack :as unpack]]
            [datomic.api :as datomic]
            [hara.event :refer [raise]])
  (:import spirit.io.datomic.types.Datomic))

(defn wrap-merge-results
  "merges results of all previous steps
 
   (-> ((wrap-merge-results gen-query
                            #(set (map %1 %2)))
        (-> (scaffold/account-db)
            (assoc-in [:process :input] [{:account/name \"A\"}
                                         {:account/name \"C\"}])))
       :process
      :emitted)
   => #{'[:find ?self :where [?self :account/name \"A\"]]
        '[:find ?self :where [?self :account/name \"C\"]]}"
  {:added "0.9"}
  ([f] (wrap-merge-results f mapcat))
  ([f merge-emit-fn]
   (fn [datasource]
     (let [processes (->> (-> datasource :process :input)
                          (map #(f (assoc-in datasource [:process :input] %)))
                          (map :process))
           inputs   (->> processes
                         (map #(dissoc % :emitted))
                         (apply merge-with
                                conj
                                {:input []
                                 :normalised []
                                 :analysed []
                                 :reviewed []
                                 :characterised []}))
           emitted  (merge-emit-fn :emitted processes)]
       (assoc datasource :process (assoc inputs :emitted emitted))))))

(defn wrap-query-keyword
  "allows queries using keyword
 
   (-> ((wrap-query-keyword gen-query)
        (-> (scaffold/account-db)
            (assoc-in [:process :input] :account/name)))
       :process
       :emitted)
   => (contains-in [:find '?self :where ['?self :account/name symbol?]])
 
   ((wrap-query-keyword gen-query)
    (-> (scaffold/account-db)
        (assoc-in [:options :ban-underscores] true)
        (assoc-in [:process :input] :account/name)))
   => throws"
  {:added "0.9"}
  [f]
  (fn [datasource]
    (let [data (-> datasource :process :input)]
      (if (keyword? data)
        (if (-> datasource :options :ban-underscores)
          (raise [:keyword-banned {:data data}])
          (f (assoc-in datasource [:process :input] {data '_})))
        (f datasource)))))

(defn wrap-query-set
  "allows a set of values to be used as queries
 
   (-> ((wrap-query-set gen-query)
        (-> (scaffold/account-db)
            (assoc-in [:process :input] #{{:account/name \"chris\"}
                                          {:account/age 9}})))
       :process
       :emitted)
  => #{'[:find ?self :where [?self :account/name \"chris\"]]
        '[:find ?self :where [?self :account/age 9]]}"
  {:added "0.9"}
  [f]
  (fn [datasource]
    (let [data (-> datasource :process :input)]
      (if (set? data)
        ((wrap-merge-results f #(set (map %1 %2))) datasource)
        (f datasource)))))

(defn wrap-query-data
  "allows additional checks on the query data, supporting ids
 
   (-> ((wrap-query-data gen-query)
        (-> (scaffold/account-db)
            (assoc-in [:process :input] 0)))
       :process
       :emitted)
   => '(0)
 
   ((wrap-query-data gen-query)
    (-> (scaffold/account-db)
        (assoc-in [:process :input] 0)
        (assoc-in [:options :ban-ids] true)))
   => throws"
  {:added "0.9"}
  [f]
  (fn [datasource]
    (let [data (-> datasource :process :input)]
      (cond (long? data)
            (if (or (-> datasource :options :ban-ids)
                    (-> datasource :options :ban-top-id))
              (raise [:id-banned {:data data}])
              (-> datasource
                  (update-in [:process]
                             #(apply assoc % (zipmap [:normalised :analysed :reviewed :characterised]
                                                     (repeat data))))
                  (assoc-in [:process :emitted] (list data))))

            (hash-map? data)
            (f datasource)

            :else
            (error "WRAP_QUERY_DATA: hash-map and long only: " data)))))

(defn gen-query
  "command for generating datalog query
 
   (-> (scaffold/account-db)
       (assoc-in [:process :input] {:account/name \"chris\"})
       (gen-query)
       :process :emitted)
   => '[:find ?self :where [?self :account/name \"chris\"]]"
  {:added "0.9"}
  [datasource]
  (-> datasource
      (update-in [:options]
                 dissoc
                 :schema-required
                 :schema-restrict
                 :schema-defaults)
      (assoc :command :query)
      (pipeline/normalise)
      (pack/pack)
      (emit/emit)))

(defn wrap-pull-raw
  "if `:raw` flag, returns only the emitted results
 
   ((wrap-pull-raw identity)
    {:options {:raw :true}
     :process {:emitted [1 2 3]}})
   => [1 2 3]
 
   ((wrap-pull-raw identity)
    {:process {:emitted [1 2 3]}})
   => {:process {:emitted [1 2 3]}}"
  {:added "0.9"}
  [f]
  (fn [datasource]
    (if (and (not (-> datasource :options :debug))
             (-> datasource :options :raw))
      (-> datasource :process :emitted)
      (f datasource))))

(defn wrap-pull-first
  "returns the first result if `:first` flag is set
 
   ((wrap-pull-first :data)
    {:data [1 2 3]
     :options {:first true}})
   => 1
 
   ((wrap-pull-first :data)
    {:data [1 2 3]})
   => #{1 2 3}"
  {:added "0.9"}
  [f]
  (fn [datasource]
    (let [result (f datasource)]
      (if (instance? Datomic result)
        result
        (if (-> datasource :options :first)
          (first result)
          (set result))))))

(defn wrap-pull-entities
  "returns the pulled data as entities
 
   (-> (doto (scaffold/account-db)
         (datomic/insert! [{:account/name \"A\"}
                           {:account/name \"B\"}]))
       (prepare/prepare-db)
       (assoc-in [:process :emitted] '#{[:find ?self :where [?self :account/name \"A\"]]
                                        [:find ?self :where [?self :account/name \"C\"]]})
      ((wrap-pull-entities select-base))
       :result
       :entities
       (->> (map :db/id)))
   => '(17592186045420)"
  {:added "0.9"}
  [f]
  (fn [datasource]
    (let [datasource (f datasource)]
      (if (-> datasource :return (= :ids))
        datasource
        (let [ids (-> datasource :result :ids)
              ents (map #(datomic/entity (:db datasource) %) ids)]
          (assoc-in datasource [:result :entities] ents))))))

(defn wrap-pull-data
  "returns the pulled data given a model
 
   (-> (doto (scaffold/account-db)
         (datomic/insert! [{:account {:name \"A\" :age 9}}
                           {:account {:name \"B\" :age 100}}]))
       (prepare/prepare-db)
       (assoc-in [:pull] {:account {:age :checked
                                    :name :unchecked}})
      (prepare/prepare-model)
       (assoc-in [:process :emitted] '[:find ?self :where [?self :account/name _]])      
       ((wrap-pull-data (wrap-pull-entities select-base)))
       :result
       :data)
   => '({:account {:age 9}, :db {:id 17592186045420}}
        {:account {:age 100}, :db {:id 17592186045421}})"
  {:added "0.9"}
  [f]
  (fn [datasource]
    (let [datasource (f datasource)]
      (if (-> datasource :return (#{:ids :entities}))
        datasource
        (let [entities (-> datasource :result :entities)
              data  (-> (map #(unpack/unpack % datasource) entities))]
          (assoc-in datasource [:result :data] data))))))

(defn wrap-pull-datasource
  "returns the datasource with pulled data attached
 
   ((wrap-pull-datasource identity)
    {:return :data
     :result {:data [1 2 3]}})
   => [1 2 3]
 
   ((wrap-pull-datasource identity)
    {:options {:debug true}
     :return :data
     :result {:data [1 2 3]}})
   => {:options {:debug true}, :return :data, :result {:data [1 2 3]}}"
  {:added "0.9"}
  [f]
  (fn [datasource]
    (let [datasource (f datasource)]
      (if (-> datasource :options :debug)
        datasource
        (let [ret (or (:return datasource) :data)]
          (get-in datasource [:result ret]))))))

(defn select-base
  "raw select command without wrappers, sets only `:ids`
 
   (-> (doto (scaffold/account-db)
         (datomic/insert! [{:account/name \"A\"}
                           {:account/name \"B\"}]))
       (prepare/prepare-db)
       (assoc-in [:process :emitted] '#{[:find ?self :where [?self :account/name \"A\"]]
                                        [:find ?self :where [?self :account/name \"C\"]]})
      (select-base)
       :result
       :ids)
   => '(17592186045420)"
  {:added "0.9"}
  [datasource]
  (let [qry (-> datasource :process :emitted)
        pull-fn (fn [qry]
                    (if (list? qry) qry
                        (->> (datomic/q qry (:db datasource))
                             (map first))))
        results (if (set? qry)
                  (mapcat pull-fn qry)
                  (pull-fn qry))]
    (assoc-in datasource [:result :ids] results)))

(defn select
  "select command including wrappers
 
   (-> (doto (scaffold/account-db)
         (datomic/insert! [{:account/name \"A\"}
                           {:account/name \"B\"}]))
       (select :account/name {:options {:ids false}}))
   => #{{:account {:name \"B\"}} {:account {:name \"A\"}}}
   
   (-> (doto (scaffold/account-db)
         (datomic/insert! [{:account/name \"A\"}
                           {:account/name \"B\"}]))
       (select :account/name {:return :ids}))
   => #{17592186045420 17592186045421}
 
   (-> (doto (scaffold/account-db)
         (datomic/insert! [{:account/name \"A\"}
                           {:account/name \"B\"}]))
       (select :account/name {:return :entities}))
   ;;=> #{{:db/id 17592186045420} {:db/id 17592186045421}}
   "
  {:added "0.9"}
  [datasource data opts]
  (let [query-fn  (-> gen-query
                      (wrap-query-data)
                      (wrap-query-keyword)
                      (wrap-query-set))
        pull-fn (-> select-base
                    (wrap-pull-entities)
                    (wrap-pull-data)
                    (wrap-pull-datasource)
                    (wrap-pull-first)
                    (wrap-pull-raw))]
    (-> datasource
        (prepare/prepare opts data)
        (assoc-nil :op :select)
        query-fn
        pull-fn)))

(defn query
  "query command for datalog style queries
 
   (-> (doto (scaffold/account-db)
         (datomic/insert! [{:account/name \"A\"}
                           {:account/name \"B\"}]))
       (query '[:find ?self :where [?self :account/name _]] [] {:options {:ids false}}))
   => #{{:account {:name \"B\"}} {:account {:name \"A\"}}}
   
   (-> (doto (scaffold/account-db)
         (datomic/insert! [{:account/name \"A\"}
                           {:account/name \"B\"}]))
       (query '[:find ?e 
                :in $ ?name
                :where [?e :account/name ?name]]
              [\"A\"]
              {:options {:ids false}}))
   => #{{:account {:name \"A\"}}}"
  {:added "0.9"}
  [datasource data qargs opts]
  (let [pull-fn  (-> (fn [datasource]
                       (assoc-in datasource [:result :ids]
                                 (map first (apply datomic/q data (:db datasource) qargs))))
                     (wrap-pull-entities)
                     (wrap-pull-data)
                     (wrap-pull-datasource)
                     (wrap-pull-first)
                     (wrap-pull-raw))]
    (-> datasource
        (prepare/prepare opts data)
        (assoc :op :query)
        pull-fn)))
