(ns spirit.io.datomic.api.transaction
  (:require [hara.common
             [checks :refer [hash-map? long?]]
             [error :refer [error]]]
            [hara.data
             [map :refer [assoc-nil]]
             [nested :refer [merge-nil-nested
                             merge-nested]]]
            [spirit.io.datomic.data :as data]
            [spirit.io.datomic.data.checks :as checks]
            [spirit.io.datomic.api
             [prepare :as prepare]
             [select :as select]
             [model :as model]
             [link :as link]
             [depack :as depack]]
            [spirit.io.datomic.process
             [pipeline :as pipeline]
             [pack :as pack]
             [unpack :as unpack]
             [emit :as emit]]
            [hara.event :refer [raise]]
            [datomic.api :as datomic]
            [clojure.walk :as walk]))

(defn gen-datoms
  "generates datums given an input
 
   (-> (scaffold/account-db)
       (assoc-in [:process :input] {:account {:name \"Chris\"
                                              :age 3
                                              :sex :m}})
       (assoc-in [:tempids] (atom #{}))
       (gen-datoms)
      :process
       :emitted)
   => (contains-in
       [{:account/name \"Chris\",
         :account/age 3,
         :account/sex
         :account.sex/m,
         :db/id checks/db-id?}])"
  {:added "0.9"}
  [datasource]
  (-> datasource
      (assoc :command :datoms)
      (pipeline/normalise)
      (pack/pack)
      (emit/emit)))

(defn wrap-datom-vector
  "allows datoms to be input as a vector
 
   (-> ((wrap-datom-vector gen-datoms)
        (-> (scaffold/account-db)
            (assoc :tempids (atom #{}))
            (assoc-in [:process :input] [{:account/name \"A\"}
                                         {:account/name \"C\"}])))
       :process
      :emitted)
   => (contains-in [{:account/name \"A\", :db/id checks/db-id?}
                    {:account/name \"C\", :db/id checks/db-id?}])"
  {:added "0.9"}
  [f]
  (fn [datasource]
    (let [data (-> datasource :process :input)]
      (if (vector? data)
        ((select/wrap-merge-results f mapcat) datasource)
        (f datasource)))))

(defn has-hash?
  "contains the `:#` entry
 
   (has-hash? {:# {:id :hello}})
   => true"
  {:added "0.9"}
  [m]
  (boolean (and (hash-map? m)
                (:# m))))

(defn replace-hash
  "constructs a :db/id from hash
 
   (replace-hash {:# {:id :hello}} {})
   => {:db {:id :hello}}
 
   (replace-hash {:# {:id 'a}} {'a (data/iid -11)})
   => {:db {:id (data/iid -11)}}"
  {:added "0.9"}
  [m ids]
  (let [id (-> m :# :id)
        id-num (cond (symbol? id) (get ids id)
                     (checks/db-id? id) (get ids (data/iid-seed id))
                     :else id)]
    (-> m
        (dissoc :#)
        (assoc-in [:db :id] id-num))))

(defn replace-ids
  "replace ids with given lookup table
 
   (replace-ids {:account/user \"A\" :# {:nss #{:account} :id '?1}
                 :account/orders #{{:order/number 2 :# {:nss #{:order} :id '?2}
                                    :order/items #{{:item/name \"D\" :# {:nss #{:item} :id '?3}}
                                                   {:item/name \"C\" :# {:nss #{:item} :id '?4}}}}
                                   {:order/number 1 :# {:nss #{:order} :id '?5}
                                    :order/items #{{:item/name \"B\" :# {:nss #{:item} :id '?6}}
                                                  {:item/name \"A\" :# {:nss #{:item} :id '?7}}}}}}
                '{?1 1
                  ?2 2
                  ?3 3
                  ?4 4
                  ?5 5
                  ?6 6
                  ?7 7})
   => {:account/user \"A\" :db {:id 1}
       :account/orders #{{:order/number 2 :db {:id 2}
                          :order/items #{{:item/name \"C\" :db {:id 4}}
                                         {:item/name \"D\" :db {:id 3}}}}
                         {:order/number 1 :db {:id 5}
                          :order/items #{{:item/name \"A\" :db {:id 7}}
                                         {:item/name \"B\" :db {:id 6}}}}}}"
  {:added "0.9"}
  [form ids]
  (walk/prewalk
   (fn [form]
     (cond (has-hash? form) (replace-hash form ids)
           (checks/db-id? form) (get ids (data/iid-seed form))
           :else form))
   form))

(defn resolve-tempids
  "resolves the previously generated tempids to actual datomic ids
 
   (let [ds (-> (scaffold/order-db)
                (datomic/insert! {:account {:user \"A\"}} :transact :datomic :debug))]
     (resolve-tempids ds (get-in ds [:result :transact])))
   => {:account {:user \"A\"}, :db {:id 17592186045418}}"
  {:added "0.9"}
  [datasource result]
  (let [ids (->> (-> datasource :tempids deref)
                 (map #(vector (data/iid-seed %)
                               (datomic/resolve-tempid (datomic/db (:connection datasource))
                                                       (:tempids result)
                                                       %)))
                 (into {}))]
    (depack/depack (replace-ids (-> datasource :process :reviewed) ids)
                   datasource)))

(defn wrap-transact-options
  "switches betwwen `:transact` values to return various results
 
   (-> (scaffold/order-db)
       (datomic/insert! {:account {:user \"A\"}} :transact :async))
   => java.util.concurrent.Future 
 
   (-> (scaffold/order-db)
       (datomic/insert! {:account {:user \"A\"}} :transact :datasource))
   => spirit.io.datomic.types.Datomic
   
   (-> (scaffold/order-db)
       (datomic/insert! {:account {:user \"A\"}} :transact :datomic)
       keys
       sort)
   => (list :db-after :db-before :tempids :tx-data)
 
   (-> (scaffold/order-db)
       (datomic/insert! {:account {:user \"A\"}} :transact :resolve))
   => {:account {:user \"A\"}, :db {:id 17592186045418}}"
  {:added "0.9"}
  [f]
  (fn [datasource]
    (let [datasource (f datasource)
          opt-trans  (or (-> datasource :transact) :resolve)
          result (cond (#{:async :promise} opt-trans)
                       (-> datasource :result :promise)

                       (#{:resolve :datomic :datasource} opt-trans)
                       (let [transact (or (-> datasource :result :simulation)
                                          @(-> datasource :result :promise))]
                         (case opt-trans
                           :datomic transact
                           :datasource (dissoc datasource :db)
                           :resolve (resolve-tempids datasource transact)))
                       
                       :else
                       (error "WRAP_TRANSACT_OPTIONS: #{:datomic :async :promise :resolve}"))]
      (assoc-in datasource [:result :transact] result))))

(defn wrap-transact-results
  "returns different results for `:transact` options
 
   (-> (scaffold/order-db)
       (datomic/insert! {:account {:user \"A\"}} :transact :datasource :debug)
       :result
       :transact)
   => spirit.io.datomic.types.Datomic"
  {:added "0.9"}
  [f]
  (fn [datasource]
    (let [datasource (f datasource)
          opt-trans  (or (-> datasource :transact) :resolve)]

      (cond (-> datasource :options :debug)
            datasource

            (#{:resolve :async :promise :datomic :datasource} opt-trans)
            (get-in datasource [:result :transact])
            
            :else
            (error "WRAP_TRANSACT_RESULTS: " opt-trans)))))

(defn transact-base
  "baseline transact function
 
   (-> (scaffold/order-db)
       (assoc :command :datoms,
              :type :datomic,
              :op :insert,
              :transact :resolve
              :tempids (atom #{})
             :process {:emitted [{:account/user \"A\", :db/id (data/iid 'id_0)}]})
       (transact-base)
       :result :promise deref keys sort vec)
   => [:db-after :db-before :tempids :tx-data]"
  {:added "0.9"}
  [datasource]
  (let [opt-trans  (-> datasource :transact)
        [datomic-fn sel-key res-key]
        (cond (-> datasource :simulate)
              [datomic/with :db :simulation]

              :else
              [(if (= opt-trans :async)
                 datomic/transact-async
                 datomic/transact)
               :connection :promise])]
    (let [datoms (-> datasource :process :emitted)
          result (datomic-fn (get datasource sel-key) datoms)]
      (assoc-in datasource [:result res-key] result))))

(defn prepare-tempids
  "creates tempids having the necessary db ids
 
   (-> (prepare-tempids {:process {:input {:user {:account (data/iid 'id_0)}}}
                         :tempids (atom #{})})
       :tempids deref)
   => #{(data/iid 'id_0)}"
  {:added "0.9"}
  [datasource]
  (let [input (-> datasource :process :input)
        review (walk/postwalk
                (fn [form]
                  (cond (checks/db-id? form)
                        (do (swap! (:tempids datasource) conj form)
                            form)
                        (and (hash-map? form)
                             (:db/id form))
                        (let [id (:db/id form)]
                          (-> form
                              (dissoc :db/id)
                              (assoc-in [:db :id] id)))
                        :else form))
                input)]
    (update-in datasource [:process] assoc :reviewed review :emitted input)))

(def transact-fn
  (-> transact-base
      (wrap-transact-options)
      (wrap-transact-results)
      (select/wrap-pull-raw)))

(defn transact!
  "transaction function for raw datoms
 
   ;; does not resolve temp ids
   (-> (scaffold/account-db)
       (transact! [{:account/name \"A\"}] {:transact :datomic})
       :tempids count)
   => 1"
  {:added "0.9"}
  [datasource datoms opts]
  (-> datasource
      (prepare/prepare opts datoms)
      (prepare-tempids)
      (assoc-nil :op :transact)
      transact-fn))

(defn insert!
  "insert function for nested data
 
   (-> (scaffold/account-db)
       (insert! [{:account/name \"A\"}
                 {:account/name \"B\"}] {}))
   => (contains-in [{:account {:name \"A\"}, :db {:id number?}}
                    {:account {:name \"B\"}, :db {:id number?}}])"
  {:added "0.9"}
  [datasource data opts]
  (let [opts (merge-nil-nested opts {:options {:schema-restrict true
                                               :schema-required true
                                               :schema-defaults true}})
        datom-fn    (-> gen-datoms
                        (wrap-datom-vector))]
    (-> datasource
        (prepare/prepare opts data)
        (assoc-nil :op :insert)
        datom-fn
        transact-fn)))

(defn wrap-delete-results
  "helper function for `delete!`, gets the ids of the elements deleted"
  {:added "0.9"}
  ([f] (wrap-delete-results f :db-before))
  ([f db-state]
     (fn [datasource]
        (let [datasource (f datasource)
              opt-trans (-> datasource :transact)
              result  (-> datasource :result :transact)]
          (if (or (nil? opt-trans)
                  (= :resolve opt-trans))
            (let [ids (set (map second result))
                  struct  (or (-> datasource :result :simulation)
                              @(-> datasource :result :promise))]
              (select/select (assoc datasource :db (db-state struct))
                             ids {:options {:debug false
                                            :ban-ids false
                                            :ban-top-id false
                                            :ids true}}))
            result)))))

(defn delete!
  "enables deletion of matching queries
 
   (-> (scaffold/account-db)
       (insert! [{:account/name \"A\"}
                 {:account/name \"B\"}] {:transact :datasource})
       ((juxt #(delete! % {:account/name \"A\"}   {:transact :resolve})
              #(datomic/select % :account/name))))
   => [#{{:account {:name \"A\"}, :db {:id 17592186045420}}}
       #{{:account {:name \"B\"}, :db {:id 17592186045421}}}]
 
   (-> (scaffold/order-db)
       (insert! [{:account {:user \"A\"
                             :orders [{:number 1}
                                      {:number 2}]}}
                 {:account {:user \"B\"
                             :orders [{:number 3}
                                      {:number 4}]}}] {:transact :datasource})
       (delete! {:account/user \"A\"} {})
      (datomic/select :order/number :ids false))
   => #{{:order {:number 1}}
        {:order {:number 2}}
        {:order {:number 3}}
        {:order {:number 4}}}
 
   "
  {:added "0.9"}
  [datasource data opts]
  (let [ids (select/select datasource data
                           (merge-nested opts {:options {:first false
                                                         :raw false}
                                               :return :ids}))
        transact-fn   (-> transact-base
                          (wrap-transact-options)
                          (wrap-delete-results)
                          (select/wrap-pull-raw))
        datoms (map (fn [x] [:db.fn/retractEntity x]) ids)]
    (-> datasource
        (prepare/prepare opts datoms)
        (prepare-tempids)
        (assoc :op :delete)
        transact-fn)))


(defn update!
  "updates data based on query
 
   (-> (scaffold/account-db)
       (insert! [{:account/name \"A\"}
                 {:account/name \"B\"}] {:transact :datasource})
       (update! {:account/name \"A\"}
                {:account {:name \"C\" :age 10}} {})
       (datomic/select :account/name :ids false))
  => #{{:account {:name \"C\", :age 10}}
        {:account {:name \"B\"}}}"
  {:added "0.9"}
  [datasource data update opts]
  (let [ids (select/select datasource data
                           (merge-nested opts {:options {:first false
                                                         :raw false}
                                               :return :ids}))
        updates (mapv (fn [id] (assoc-in update [:db :id] id)) ids)
        datasource (if (-> datasource :options :ban-ids)
              (-> datasource
                  (assoc-in [:options :ban-ids] false)
                  (assoc-in [:options :ban-body-ids] true))
              datasource)]
    (-> datasource
        (assoc :op :modify)
        (insert! updates (merge-nil-nested
                          opts {:options {:schema-required false
                                          :schema-defaults false}})))))

(defn delete-all!
  "
 
   (-> (doto (scaffold/order-db)
          (insert! [{:account {:user \"A\"
                               :orders [{:number 1}
                                        {:number 2}]}}
                    {:account {:user \"B\"
                               :orders [{:number 3}
                                       {:number 4}]}}] {}))
       (delete-all! {:account/user \"A\"} {:access {:account {:orders :checked}}}))
   => (contains-in #{{:account {:orders #{{:number 2, :+ {:db {:id number?}}}
                                          {:number 1, :+ {:db {:id number?}}}}, :user \"A\"},
                      :db {:id number?}}})"
  {:added "0.9"}
  [datasource data opts]
  (let [sdatasource (select/select datasource data
                                   (-> opts
                                       (merge-nested {:options {:first false
                                                                :raw false
                                                                :debug true}
                                                      :return :entities})
                                       (update-in [:pipeline] dissoc :pull)))
        entities (-> sdatasource :result :entities)
        ret-model (if-let [imodel (-> sdatasource :pipeline :allow)]
                    (model/model-unpack imodel (-> sdatasource :schema :tree))
                    (raise :missing-allow-model))
        all-ids (mapcat (fn [entity]
                          (link/linked-ids entity ret-model
                                           (-> sdatasource :schema :flat)))
                        entities)
        transact-fn   (-> transact-base
                          (wrap-transact-options)
                          (wrap-delete-results)
                          (select/wrap-pull-raw))
        datoms (map (fn [x] [:db.fn/retractEntity x]) all-ids)
        result (-> datasource
                   (prepare/prepare (merge opts {:transact :datomic}) datoms)
                   (prepare-tempids)
                   (assoc :op :delete)
                   transact-fn)]
    (let [opt-trans (-> datasource :transact)]
      (cond (or (nil? opt-trans)
                (= :resolve opt-trans))
            (select/select (assoc datasource :db (:db-before result))
                           (set (map :db/id entities))
                           (merge opts {:options {:debug false
                                                  :ban-ids false
                                                  :ban-top-id false
                                                  :ids true}}))

            :else
            (= :datomic opt-trans) result

            :else ("DELETE-ALL!: Options for :transact are #{:resolve(default) :transact}")
        result))))
