(ns spirit.io.datomic.process.emit.characterise
  (:require [hara.common.error :refer [error]]
            [hara.common.checks :refer [hash-map? long?]]
            [hara.data.nested :refer [merge-nested]]
            [spirit.io.datomic.data :refer [iid isym]]
            [spirit.io.datomic.data.checks :refer [db-id?]]))

(defn wrap-gen-id
  "converts the [:# :id] entry from a symbol to a db id
   
   ((wrap-gen-id (fn [x _ _] x))
    {:animal/name \"Lion\" :# {:nss #{:node :animal} :id '?id_0}
     :node/key \"A\"}
    {:characterise characterise}
    {:tempids (atom #{})})
   => (contains-in {:animal/name \"Lion\"
                    :node/key \"A\"
                    :# {:nss #{:node :animal}
                        :id checks/db-id?}}) "
  {:added "0.9"}
  [f]
  (fn [pdata fns datasource]
    (let [id  (get-in pdata [:# :id])
          nid (cond (nil? id)
                    (let [gen  (-> datasource :options :generate-ids)]
                      (if (fn? gen) (gen) (iid)))

                    (symbol? id) (iid id)

                    :else id)]
      (swap! (:tempids datasource) conj nid)
      (f (assoc-in pdata [:# :id] nid) fns datasource))))

(defn wrap-gen-sym
  "generates a symbol for the entry, used for query generation
 
   ((wrap-gen-sym (fn [x _ _] x))
    {:animal/name \"Lion\" :# {:nss #{:node :animal} :id '?id_0}
     :node/key \"A\"}
    {:characterise characterise}
    {:options {:generate-syms true}})
   => (contains-in {:# {:nss #{:node :animal}
                       :id symbol?
                        :sym symbol?}})"
  {:added "0.9"}
  [f]
  (fn [pdata fns datasource]
    (if (nil? (get-in pdata [:# :sym]))
      (let [gen    (-> datasource :options :generate-syms)
            sym (if (fn? gen) (gen) (isym))]
        (assoc-in (f pdata fns datasource) [:# :sym] sym))
      (f pdata fns datasource))))

(defn wrap-reverse
  "adds the :rid and :rkey to the entry
 
   ((wrap-reverse identity
                   true
                   :RID
                   {:type :ref
                    :ref {:rkey :node/parent}}
                   {})
   {:animal/name \"Tiger\"
     :node/key \"B\"
     :# {:nss #{:node :animal} :id :ID}})
   => {:animal/name \"Tiger\"
       :node/key \"B\"
       :# {:nss #{:node :animal}
           :id :ID
           :rid :RID
           :rkey :node/parent}}"
  {:added "0.9"}
  [f is-reverse pid attr datasource]
  (fn [x]
    (let [res (f x)]
      (if is-reverse
        (-> res
            (assoc-in [:# :rid] pid)
            (assoc-in [:# :rkey] (-> attr :ref :rkey)))
        res))))

(defn characterise-ref-single
  "creates a structure characterising a given node for refs of cardinality :one
 
   (characterise-ref-single :image/item
                            {:item/name \"hello\" :# {:nss #{:item} :id '?id_0}}
                            (data/iid '?id_0)
                            {:type :ref
                             :cardinality :one
                             :ref {:ns :item
                                  :rval :images
                                   :type :forward
                                   :key :image/item
                                   :val :item
                                   :rkey :image/_item
                                   :rident :item/images}
                             :ident :image/item}
                            {:characterise (wrap-gen-id characterise-loop)}
                            {:tempids (atom #{})
                             :schema (schema/schema examples/account-orders-items-image)}        
                            {})
   => {:refs-one {:image/item {:data-one {:item/name \"hello\"}
                               :# {:nss #{:item}
                                   :id (data/iid '?id_0)}}}}"
  {:added "0.9"}
  [k v pid attr fns datasource output]
  (let [is-reverse (not= k (-> attr :ref :key))
        lu         (if is-reverse (-> attr :ref :rkey) (-> attr :ref :key))
        cat-data   (if is-reverse :revs-one :refs-one)
        cat-id     (if is-reverse :rev-ids-one :ref-ids-one)]
    (cond (or (long? v) (db-id? v) (symbol? v))
          (assoc-in output
                    [(if is-reverse :rev-ids :ref-ids) lu]
                    #{v})

          (hash-map? v)
          (let [id   (get-in v [:# :id])
                f    (wrap-reverse #((:characterise fns) % fns datasource)
                                   is-reverse pid attr datasource)
                res  (f v)
                output (assoc-in output [cat-data lu] res)]
            output))))

(defn characterise-ref-many
  "creates a structure characterising a given node for refs of cardinality :many
   
   (characterise-ref-many :node/children
                          #{{:animal/name \"Tiger\"
                             :node/key \"B\"
                             :# {:nss #{:node :animal} :id '?id_0}}}
                          (data/iid '?id_0)
                          {:ident :node/children
                           :cardinality :many
                           :type :ref :ref {:ns :node
                                            :type :reverse
                                             :val :children
                                             :key :node/_parent
                                             :rval :parent
                                             :rkey :node/parent
                                             :rident :node/parent}}
                          {:characterise (wrap-gen-id characterise-loop)}
                          {:tempids (atom #{})
                           :schema (schema/schema examples/node-animal-plant)}
                          {})
   => {:revs-many {:node/parent #{{:data-one {:animal/name \"Tiger\" :node/key \"B\"}
                                   :# {:nss #{:node :animal}
                                       :id (data/iid '?id_0)
                                       :rid (data/iid '?id_0)
                                       :rkey :node/parent}}}}}"
  {:added "0.9"}
  [k vs pid attr fns datasource output]
  (let [is-reverse  (not= k (-> attr :ref :key))
        lu         (if is-reverse (-> attr :ref :rkey) (-> attr :ref :key))
        cat-data   (if is-reverse :revs-many :refs-many)
        cat-id  (if is-reverse :rev-ids-many :ref-ids-many)
        id-pred  (fn [x] (or (long? x) (db-id? x) (symbol? x)))
        all-maps (filter (complement id-pred) vs)
        map-ids  (->> all-maps
                      (map #(get-in % [:# :id])))
        all-ids  (->> vs
                      (filter id-pred))
        output   (if (empty? all-ids) output
                     (-> output
                         (assoc-in [cat-id lu]
                                   (set (filter identity map-ids)))
                         (assoc-in [(if is-reverse :rev-ids :ref-ids) lu]
                                   (set all-ids))))
        output   (if (empty? all-maps) output
                     (assoc-in output [cat-data lu]
                               (set (map (wrap-reverse
                                          #((:characterise fns) % fns datasource)
                                          is-reverse pid attr datasource)
                                         all-maps))))]
    output))

(defn characterise-entry
  "creates a structure characterising a given node for all data
 
   (characterise-entry :account/age
                       9
                       (data/iid '?id_0)
                       {:characterise (wrap-gen-id characterise-loop)}
                       
                       {:tempids (atom #{})
                        :schema (schema/schema examples/account-name-age-sex)}
                       {:data-one {:account/name \"Chris\"}})
  => {:data-one {:account/name \"Chris\", :account/age 9}}
   "
  {:added "0.9"}
  [k v pid fns datasource output]
  (let [[attr] (-> datasource :schema :flat k)
        t (:type attr)]
    (if (and attr t)
      (cond (list? v) (assoc-in output [:db-funcs k] v)

            (and (set? v) (get v '_)) (assoc-in output [:data-many k] #{'_})

            (not= :ref t)
            (cond
             (set? v)  (assoc-in output [:data-many k] v)
             :else     (assoc-in output [:data-one k] v))

            (= :ref t)
            (cond (set? v) (characterise-ref-many k v pid attr fns datasource output)
                  :else (characterise-ref-single k v pid attr fns datasource output)))

      (cond (= k :db)
            (assoc output k v)

            (= k :#)
            (assoc output k (merge-nested (output k) v))

            :else
            (error "key " k " not found in schema.")))))


(defn characterise-loop
  "characterises elements in the main loop
 
   (characterise-loop {:account/name \"Chris\"
                       :account/age 9
                       :account/sex :account.sex/m
                       :# {:nss #{:account}
                           :id (data/iid '?id_0)}}
                      {:characterise (wrap-gen-id characterise-loop)}
                     {:tempids (atom #{})
                       :schema (schema/schema examples/account-name-age-sex)})
   => {:data-one {:account/name \"Chris\"
                  :account/age 9
                  :account/sex
                  :account.sex/m}
       :# {:nss #{:account}
           :id (data/iid '?id_0)}}"
  {:added "0.9"}
  ([pdata fns datasource]
   (let [id (get-in pdata [:# :id])]
     (characterise-loop pdata id fns datasource {})))
  ([pdata id fns datasource output]
   (if-let [[k v] (first pdata)]
     (recur (next pdata) id fns datasource
            (characterise-entry k v id fns datasource output))
     output)))

(defn characterise-raw
  "basic characterise function
   
   (characterise-raw {:image/item {:item/name \"hello\",
                                   :# {:nss #{:item}, :id '?id_0}},
                      :image/url \"www.image\",
                      :# {:nss #{:image} :id '?id_1}}
                     {:tempids (atom #{})
                      :schema (schema/schema examples/account-orders-items-image)})
   => {:refs-one {:image/item {:data-one {:item/name \"hello\"},
                               :# {:nss #{:item}, :id '?id_0}}},
      :data-one {:image/url \"www.image\"},
       :# {:nss #{:image}, :id '?id_1}}"
  {:added "0.9"}
  [pdata datasource]
  (let [fns {:characterise
             (let [f characterise-loop
                   f (cond (and (= :query (:command datasource))
                                (not (false? (-> datasource :options :generate-syms))))
                           (wrap-gen-sym f)

                           (and (= :datoms (:command datasource))
                                (not (false? (-> datasource :options :generate-ids))))
                           (wrap-gen-id f)

                           :else f)]
               f)}]
    ((:characterise fns) pdata fns datasource)))

(defn characterise
  "top level characterise function
   
   (-> (scaffold/node-db {:name \"characterise-test\"})
       (assoc-in [:process :reviewed]
                 {:node/key \"A\",
                  :node/children #{{:animal/name \"Tiger\",
                                    :# {:nss #{:node :animal},
                                        :id '?id_0},
                                    :node/key \"B\"}},
                  :# {:nss #{:node}, :id '?id_1}})
      (characterise)
       :process
       :characterised)
   => {:data-one {:node/key \"A\"},
       :revs-many {:node/parent #{{:data-one {:animal/name \"Tiger\",
                                              :node/key \"B\"},
                                   :# {:nss #{:node :animal},
                                       :id '?id_0,
                                       :rid '?id_1,
                                       :rkey :node/parent}}}},
       :# {:nss #{:node}, :id '?id_1}}"
  {:added "0.9"}
  [datasource]
  (let [data (-> datasource :process :reviewed)
        ndata (characterise-raw data datasource)]
    (assoc-in datasource [:process :characterised] ndata)))
