(ns spirit.io.datomic.process.emit.datoms
  (:require [spirit.io.datomic.data.checks :refer [db-id?]]
            [hara.common.error :refer [error]]))

(def ^:dynamic *nested* nil)

(defn datom-funcs
  "creates datomic functions from inputs
 
   (datom-funcs {:db-funcs {:account/name '(:get-name \"USER\")}
                 :data-one {:account/age 9}
                 :# {:nss #{:account}
                     :id (data/iid '?id_0)}})
   => [[:get-name (data/iid '?id_0) :account/name \"USER\"]]"
  {:added "0.9"}
  [chd]
  (let [cid (get-in chd [:# :id])
        dbs (for [[k v] (:db-funcs chd)]
              (apply vector (keyword (first v)) (get-in chd [:# :id]) k (rest v)))]
    (concat
     (mapcat datom-funcs (vals (:refs-one chd)))
     (mapcat datom-funcs (vals (:revs-one chd)))
     (mapcat datom-funcs (apply concat (vals (:refs-many chd))))
     (mapcat datom-funcs (apply concat (vals (:revs-many chd))))
     dbs)))

(defn datom-ids
  "creates datomic structures where inputs are ids
 
   (datom-ids {:data-one {:account/user \"chris\"},
               :data-many {:account/tags #{\"hello\" \"world\"}}
               :rev-ids-many {:order/account #{}}
               :rev-ids {:order/account #{1000 2000}}
               :# {:nss #{:account}
                   :id (data/iid '?id_0)}})
  => [[:db/add 1000 :order/account (data/iid '?id_0)]
       [:db/add 2000 :order/account (data/iid '?id_0)]]"
  {:added "0.9"}
  [chd]
  (let [cid (get-in chd [:# :id])
        rfs-fn (fn [rids k]
                 (map (fn [rid] [:db/add cid k rid]) rids))
        rvs-fn (fn [rids k]
                 (map (fn [rid] [:db/add rid k cid]) rids))]
    (concat
     (mapcat datom-ids (vals (:refs-one chd)))
     (mapcat datom-ids (vals (:revs-one chd)))
     (mapcat datom-ids (apply concat (vals (:refs-many chd))))
     (mapcat datom-ids (apply concat (vals (:revs-many chd))))
     (mapcat rfs-fn (vals (:ref-ids chd)) (keys (:ref-ids chd)))
     (mapcat rvs-fn (vals (:rev-ids chd)) (keys (:rev-ids chd))))))

(defn datom-tree
  "generates datoms by walking the characterised datastructure
 
   (binding [*nested* (atom [])]
     (let [res (datom-tree {:data-one {:node/key \"A\"}
                            :revs-many {:node/parent #{{:data-one {:animal/name \"Tiger\"
                                                                   :node/key \"B\"}
                                                        :# {:nss #{:node :animal}
                                                            :id (data/iid '?id_0)
                                                            :rid (data/iid '?id_1)
                                                            :rkey :node/parent}}}}
                           :# {:nss #{:node} :id (data/iid '?id_1)}})]
       [res @*nested*]))
   
   => [[{:node/key \"A\", :db/id (data/iid '?id_1)}]
       [{:animal/name \"Tiger\", :node/key \"B\", :db/id (data/iid '?id_0)}
        {:db/id (data/iid '?id_0), :node/parent (data/iid '?id_1)}]]"
  {:added "0.9"}
  [chd]
  (let [ref-ones (map (fn [k rf]
                        (let [[trunk & rest] (datom-tree rf)]
                          [k trunk rest]))
                      (keys (:refs-one chd))
                      (vals (:refs-one chd)))
        rev-link  (if-let [rid (get-in chd [:# :rid])]
                    [{:db/id (get-in chd [:# :id]) (get-in chd [:# :rkey]) rid}]
                    [])
        trunk (-> (into {}
                        (map (fn [k rfs]
                               [k (set (mapcat datom-tree rfs))])
                             (keys (:refs-many chd))
                             (vals (:refs-many chd))))
                  (merge (into {}
                               (map (fn [[k trunk]] [k trunk]) ref-ones)))
                  (merge (:data-one chd))
                  (merge (:data-many chd)))
        tarr     (if-not (empty? trunk)
                   (let [trunk    (if-let [id (get-in chd [:# :id])]
                                    (assoc trunk :db/id id) trunk)]
                     [trunk])
                   [])
        revs     (mapcat datom-tree (apply concat (vals (:revs-many chd))))
        _        (swap! *nested* concat revs)]
    (concat
     tarr
     rev-link
     (mapcat (fn [[_ _ rest]] rest) ref-ones))))

(defn wrap-check-empty
  "if ref is empty, throw an error
 
   (-> (scaffold/account-db)
       (datomic/insert! {:account {}}))
   => throws"
  {:added "0.9"}
  [f]
  (fn [chdata]
    (if (and (db-id? (get-in chdata [:# :id]))
             (empty? (:db-funcs chdata))
             (empty? (:data-one chdata))
             (empty? (:data-many chdata))
             (empty? (:refs-one chdata))
             (empty? (:refs-many chdata)))
      (error "WRAP_CHECK_EMPTY: Cannot allow an empty ref for: " chdata)
      (f chdata))))

(defn datoms-raw
  "generates all datoms given an input
 
   (datoms-raw {:data-one {:node/key \"A\"}
                :revs-many {:node/parent #{{:data-one {:animal/name \"Tiger\"
                                                       :node/key \"B\"}
                                            :# {:nss #{:node :animal}
                                                :id (data/iid '?id_0)
                                                :rid (data/iid '?id_1)
                                               :rkey :node/parent}}}}
                :# {:nss #{:node} :id (data/iid '?id_1)}})
   => [{:node/key \"A\", :db/id (data/iid '?id_1)}
       {:animal/name \"Tiger\", :node/key \"B\", :db/id (data/iid '?id_0)}
       {:db/id (data/iid '?id_0), :node/parent (data/iid '?id_1)}]"
  {:added "0.9"}
  [chdata]
  (binding [*nested* (atom [])]
    (vec (concat
          ((wrap-check-empty datom-tree) chdata)
          @*nested*
          (datom-ids chdata)
          (datom-funcs chdata)))))

(defn datoms
  "top level datom emit function
 
   (-> (scaffold/order-db)
       (assoc :tempids (atom #{}))
       (assoc-in [:process :characterised]
                 {:refs-one {:image/item {:data-one {:item/name \"hello\"}
                                          :# {:nss #{:item}
                                              :id (data/iid '?id_0)}}}
                 :data-one {:image/url \"www.image\"}
                  :# {:nss #{:image}  :id (data/iid '?id_1)}})
       (datoms)
       :process
       :emitted)
   => [{:image/item {:item/name \"hello\"
                     :db/id (data/iid '?id_0)}
        :image/url \"www.image\" :db/id (data/iid '?id_1)}]"
  {:added "0.9"}
  [datasource]
  (let [chdata (-> datasource :process :characterised)
        ndata (datoms-raw chdata)]
    (assoc-in datasource [:process :emitted] ndata)))
