(ns spirit.io.datomic.process.emit.query
  (:require [hara.common
             [error :refer [error]]
             [checks :refer [hash-map?]]]
            [spirit.io.datomic.data :refer [isym]]))

(defn walk-replace
  "replaces the symbols with the ones in the map
 
   (walk-replace ?init
                 {'??sym 'sym '? 'esym '??attr 'k})
   => '[sym k esym]"
  {:added "0.9"}
  [st rep]
  (cond (vector? st) (mapv #(walk-replace % rep) st)
        (list? st) (map #(walk-replace % rep) st)
        (hash-map? st) (zipmap (keys st)
                               (map #(walk-replace % rep) (vals st)))
        (rep st) (rep st)
        :else st))

(defn query-sym
  "gets the symbol in the characterised structure
 
   (query-sym {:# {:sym 'id}})
   => 'id"
  {:added "0.9"}
  [chdata]
  (get-in chdata [:# :sym]))

(def ?init '[??sym ??attr ?])

(defn has-placeholder?
  "checks if a list has a placeholder
 
   (has-placeholder? '(+ 1 2)) => false
 
   (has-placeholder? '(+ 1 ?)) => true"
  {:added "0.9"}
  [lst]
  (boolean (some #(= '? %) lst)))

(defn q-fn
  "creates a datalog query
   
   (q-fn '(.contains ? \"C\"))
   => '[[??sym ??attr ?] [(.contains ? \"C\")]]"
  {:added "0.9"}
  [[x & xs :as lst]]
  (if (has-placeholder? lst)
    [?init [lst]]
    [?init [(apply list x '? xs)]]))

(defn not-fn
  "creates a negative function
 
   (not-fn \"hello\")
   => '[[??sym ??attr ?] [(not= ? \"hello\")]]"
  {:added "0.9"}
  [val]
  [?init [(list 'not= '? val)]])

(defn fulltext-fn
  "creates a fulltext search function
 
   (fulltext-fn \"hello\")
   => '[[(fulltext $ ??attr \"hello\") [[??sym ?]]]]"
  {:added "0.9"}
  [val]
  [[(list 'fulltext '$ '??attr val) [['??sym '?]]]])

(defn query-parse-list
  "supports parsing of lists
 
   (query-parse-list '(?fulltext \"hello\"))
   => '[[(fulltext $ ??attr \"hello\") [[??sym ?]]]]
 
   (query-parse-list '(?not \"hello\"))
   => '[[??sym ??attr ?] [(not= ? \"hello\")]]"
  {:added "0.9"}
  [[x & xs :as lst]]
  (cond
   (= x '?fulltext) (fulltext-fn (second lst))
   (= x '?not) (not-fn (second lst))
   :else (q-fn lst)))

(defn query-replace-fn
  "replaces elements in the query
 
   (query-replace-fn '?e_0
                     :account/name
                     '[[??sym ??attr ?]
                       [(.contains ? \"C\")]]
                     {:options {:generate-syms (fn [] '?id_1)}})
   => '[[?e_0 :account/name ?id_1]
       [(.contains ?id_1 \"C\")]]"
  {:added "0.9"}
  [sym k v datasource]
  (let [gen (-> datasource :options :generate-syms)
        symgen (if (fn? gen) gen isym)
        esym (symgen)]
    (walk-replace v {'??sym sym '? esym '??attr k})))

(defn query-data-val
  "generates query data element
 
   (query-data-val '?e_0
                   :account/name
                   '_
                   {})
   => '[[?e_0 :account/name _]]
   "
  {:added "0.9"}
  [sym k v datasource]
  (cond (list? v)
        (query-replace-fn sym k (query-parse-list v) datasource)

        :else
        [[sym k v]]))

(defn query-data
  "creates query data based on input
 
   (query-data '{:data-many {:account/name #{?e_2}},
                 :# {:sym ?self, :id ?id_2}}
               {})
   => '[[?self :account/name ?e_2]]
 
   (query-data '{:data-many {:node/key [\"A\" \"B\"]}
                 :# {:sym ?self, :id ?id_2}}
               {})
   => '[[?self :node/key \"A\"]
        [?self :node/key \"B\"]]"
  {:added "0.9"}
  [chdata datasource]
  (let [sym  (query-sym chdata)
        data (for [[k vs] (:data-many chdata)
                   v     vs]
               (query-data-val sym k v datasource))]
    (apply concat data)))

(defn query-refs
  "creates query data based on ref links
 
   (query-refs '{:revs-many {:node/parent #{{:data-many {:node/key #{\"B\"}},
                                             :# {:sym ?id_1, :rid ?id_2, :rkey :node/parent}}}},
                 :# {:sym ?self, :id ?id_2}})
   
   => '[[?id_1 :node/parent ?self]]"
  {:added "0.9"}
  [chdata]
  (let [sym (query-sym chdata)]
    (concat
     (filter identity
             (for [[k rs] (:refs-many chdata)
                   r   rs]
               (if-let [rid (get-in r [:# :id])]
                 [sym k rid])))
     (for [[k rs] (:refs-many chdata)
           r   rs]
       [sym k (query-sym r)])
     (filter identity
             (for [[k rs] (:revs-many chdata)
                   r   rs]
               (if-let [rid (get-in r [:# :id])]
                 [rid k sym])))
     (for [[k rs] (:revs-many chdata)
           r   rs]
       [(query-sym r) k sym])
     (for [[k ids] (:ref-ids chdata)
           id ids]
       [sym k id])
     (for [[k ids] (:rev-ids chdata)
           id ids]
       [id k sym]))))

(defn query-init
  "creates query data for complete input
   
   (query-init '{:data-many {:node/key #{\"B\"}},
                 :refs-many {:node/parent #{{:data-many {:node/key #{\"A\"}},
                                             :# {:sym ?id_1}}}},
                 :# {:sym ?self, :id ?id_2}}
               {})
   => '[[?self :node/key \"B\"]
        [?self :node/parent ?id_1]
        [?id_1 :node/key \"A\"]]"
  {:added "0.9"}
  [chdata datasource]
  (cond
   (nil? (seq chdata)) []
   :else
   (concat  (query-data chdata datasource)
            (query-refs chdata)
            (mapcat (fn [x]
                      (mapcat #(query-init % datasource) (second x)))
                    (concat (:refs-many chdata) (:revs-many chdata))))))

(defn query-raw
  "generates true datalog query
   
   (query-raw '{:data-many {:node/key #{\"B\"}},
                :refs-many {:node/parent #{{:data-many {:node/key #{\"A\"}},
                                            :# {:sym ?id_1}}}},
                :# {:sym ?self, :id ?id_2}}
              {})
   => '[:find ?self :where
        [?self :node/key \"B\"]
        [?self :node/parent ?id_1]
       [?id_1 :node/key \"A\"]]"
  {:added "0.9"}
  [chdata datasource]
  (let [res (query-init chdata datasource)]
    (if (empty? res)
      (error "QUERY The generated query is empty for " chdata))
    (vec (concat [:find (query-sym chdata) :where]
                 res))))

(defn query
  "top level query generation function
 
   (-> (scaffold/order-db)
       (assoc :tempids (atom #{}))
       (assoc-in [:process :characterised]
                 '{:data-many {:account/user #{\"chris\"},
                               :account/tags #{\"hello\" \"world\"}},
                   :rev-ids-many {:order/account #{}},
                  :rev-ids {:order/account #{1000 2000}},
                   :# {:sym ?self, :id ?id_0}})
       (query)
       :process
       :emitted)
   => '[:find ?self :where
        [?self :account/user \"chris\"]
        [?self :account/tags \"hello\"]
        [?self :account/tags \"world\"]
        [1000 :order/account ?self]
        [2000 :order/account ?self]]"
  {:added "0.9"}
  [datasource]
  (let [chdata (-> datasource :process :characterised)
        ndata (query-raw chdata datasource)]
    (assoc-in datasource [:process :emitted] ndata)))
