(ns spirit.io.datomic.process.pack.review
  (:require [hara.common.checks :refer [hash-map? long?]]
            [hara.common.error :refer [error]]
            [hara.string.path :as path]
            [clojure.set :as set]))

(defn find-keys
  "finds keys of a certain type
   (find-keys {:account/name [{:type :string, :cardinality :one}]
               :account/age [{:type :string, :cardinality :one}]}
              #{:account}
              :cardinality
              :one)
   => #{:account/age :account/name}"
  {:added "0.9"}
  ([fschm mk mv]
   (find-keys fschm (constantly true) mk mv))
  ([fschm nss mk mv]
   (let [comp-fn  (fn [val cmp] (if (fn? cmp) (cmp val) (= val cmp)))
         filt-fn  (fn [k] (and (nss (path/path-ns k))
                               (comp-fn (mk (first (fschm k))) mv)))]
     (set (filter filt-fn (keys fschm))))))

(defn review-required
  "throws an error if any of the keys are missing:
   
   (review-required {:# {:nss #{:account}, :account/name \"Chris\"}}
                    {:account/name [{:type :string
                                     :required true}]
                     :account/age [{:type :string
                                    :required true}]}
                    #{:account/age :account/name}
                    {})
   => throws"
  {:added "0.9"}
  [pdata fsch ks datasource]
  (if (empty? ks)
    pdata
    (error "The following keys are required: " ks)))

(defn review-defaults
  "sets the default value to the data
 
   (review-defaults {:user/name \"chris\"}
                    {:user/name [{:type :string,
                                  :cardinality :one,
                                  :ident :user/name}],
                     :user/created [{:type :string,
                                     :cardinality :one,
                                    :default (fn []
                                                \"NOW\")}]}
                    #{:user/created}
                    {})
   => {:user/name \"chris\", :user/created \"NOW\"}"
  {:added "0.9"}
  [pdata fsch ks datasource]
  (if-let [k (first ks)]
    (let [[attr] (fsch k)
          t      (:type attr)
          dft    (:default attr)
          value  (if (fn? dft) (dft) dft)
          value  (if (and (not (set? value))
                          (or (= :query (:command datasource))
                              (= :many (:cardinality attr))))
                   #{value} value)
          npdata  (cond (= t :keyword)
                        (assoc pdata k value)

                        (= t :enum)
                        (assoc pdata k (if-let [ens (-> attr :enum :ns)]
                                         (path/join [ens value])
                                         value))
                        :else
                        (assoc pdata k value))]
      (recur npdata fsch (next ks) datasource))
    pdata))

(defn expand-ns-keys
  "adds all related keys to the set
 
   (expand-ns-keys :account/age)
   => #{:account :account/age}"
  {:added "0.9"}
  ([k] (expand-ns-keys k #{}))
  ([k output]
    (if (nil? k) output
        (if-let [nsk (path/path-ns k)]
          (expand-ns-keys nsk (conj output k))
          (conj output k)))))

(defn expand-ns-set
  "combines all related keys to a single set
 
   (expand-ns-set #{:account/age
                    :user/name})
   => #{:user/name :account :account/age :user}"
  {:added "0.9"}
  ([s] (expand-ns-set s #{}))
 ([s output]
    (if-let [k (first s)]
      (expand-ns-set (next s)
                     (set/union output
                                (expand-ns-keys k)))
      output)))

(declare review-current)

(defn wrap-id
  "checks if the data is of type id
 
   (def example (scaffold/node-db))
   (-> example
       (datomic/insert! {:node {:key \"A\"
                                :children [{:key \"B\"
                                            :children [{:key \"C\"}]}]}}))
   => {:db {:id 17592186045418}
       :node {:key \"A\"
              :children #{{:+ {:db {:id 17592186045420}}
                           :key \"B\"
                           :children #{{:+ {:db {:id 17592186045419}}
                                        :key \"C\"}}}}}}
   
   (datomic/select example {:node {:parent 17592186045420}})
   => #{{:node {:key \"C\"}
         :db {:id 17592186045419}}}"
  {:added "0.9"}
  [f k]
  (fn [pdata fsch merge-fn datasource]
    (if (and (or (long? pdata) (symbol? pdata))
             (-> datasource :schema :flat k first :type (= :ref)))
      pdata
      (f pdata fsch merge-fn datasource))))

(defn review-fn
  "merges both ref and data keys for review
 
   (review-fn {:node/key \"C\" :# {:nss #{:node} :id '?id_0}}
              (-> (schema/schema examples/node-animal-plant) :flat)
              
              {:label :required :function review-required}
              {})
   => {:node/key \"C\" :# {:nss #{:node} :id '?id_0}}"
  {:added "0.9"}
  [pdata fsch merge-fn datasource]
  (let [nss   (expand-ns-set (get-in pdata [:# :nss]))
        ks    (find-keys fsch nss (-> merge-fn :label) (complement nil?))
        refks (find-keys fsch nss :type :ref)
        dataks     (try (set (keys pdata))
                        (catch Throwable t
                          (throw t)))
        mergeks    (set/difference ks dataks)
        datarefks  (set/intersection refks dataks)]
    (-> pdata
        ((-> merge-fn :function) fsch mergeks datasource)
        (review-current fsch datarefks merge-fn datasource))))

(defn review-current
  "reviews current given data and schema attribute"
  {:added "0.9"}
  [pdata fsch ks merge-fn datasource]
  (if-let [k (first ks)]
    (let [meta   (-> (fsch k) first)
          pr-fn  (fn [rf] ((wrap-id review-fn k) rf fsch merge-fn datasource))
          npdata  (if (or (= :query (:command datasource))
                          (= :many (:cardinality meta)))
                    (assoc pdata k (set (map pr-fn (pdata k))))
                    (assoc pdata k (pr-fn (pdata k))))]
      (recur npdata fsch (next ks) merge-fn datasource))
    pdata))

(defn review-raw
  "checks for required data
   
   (review-raw {:# {:nss #{:account} :account/name \"Chris\"}}
               {:schema (schema/schema {:account {:name [{:required true}]
                                                  :age  [{:required true}]}})
                :options {:schema-required true}})
   => throws"
  {:added "0.3"}
  [pdata datasource]
  (if (and (not= :query (:command datasource))
           (or (-> datasource :options :schema-defaults)
               (-> datasource :options :schema-required)))
    (let [fsch   (-> datasource :schema :flat)
          pdata  (if (-> datasource :options :schema-defaults)
                   (review-fn pdata fsch
                              {:label :default
                               :function review-defaults}
                              datasource)
                   pdata)
          pdata  (if (-> datasource :options :schema-required)
                   (review-fn pdata fsch
                              {:label :required
                               :function review-required}
                              datasource)
                   pdata)]
      pdata)
    pdata))

(defn review
  "top level review function
 
   (-> (scaffold/node-db)
       (assoc-in [:process :analysed]
                 {:node/key \"A\"
                  :node/children #{{:node/key \"B\"
                                    :node/children #{{:node/key \"C\"
                                                      :# {:nss #{:node}
                                                         :id '?id_2}}}
                                    :# {:nss #{:node}
                                        :id '?id_1}}}
                  :# {:nss #{:node}
                      :id '?id_0}})
       (review)
       :process :reviewed)
   => {:node/key \"A\"
       :node/children #{{:node/key \"B\"
                         :node/children #{{:node/key \"C\"
                                           :# {:nss #{:node} :id '?id_2}}}
                         :# {:nss #{:node} :id '?id_1}}}
       :# {:nss #{:node} :id '?id_0}}"
  {:added "0.9"}
  [datasource]
  (let [data (-> datasource :process :analysed)
        ndata (review-raw data datasource)]
    (assoc-in datasource [:process :reviewed] ndata)))
