(ns spirit.io.datomic.process.pipeline
  (:require [spirit.core.pipeline :as pipeline]
            [hara.data.path :as data]
            [hara.common.error :refer [error]]
            [hara.function.args :refer [op]]
            [spirit.io.datomic.process.pipeline
             [db :as db]
             [expression :as expression]
             [id :as id]
             [list :as list]
             [set :as set]
             [symbol :as symbol]
             [underscore :as underscore]
             [vector :as vector]
             [type-check :as type-check]]))

(def datomic-wrapper-fns
  {:underscore        underscore/wrap-branch-underscore
   :set               set/wrap-attr-set
   :vector-attr       vector/wrap-attr-vector
   :expression        expression/wrap-single-model-expression
   :db                db/wrap-db
   :id                id/wrap-single-id
   :list              list/wrap-single-list 
   :symbol            symbol/wrap-single-symbol 
   :vector-single     vector/wrap-single-vector
   :type-check        type-check/wrap-single-type-check})

(defn datomic-additions
  "additional wrappers for the normalise loop
 
   (datomic-additions {:pipeline {:expression true}})"
  {:added "0.9"}
  [{:keys [pipeline] :as datasource}]
  {:normalise              {:pre [:db]}
   :normalise-branch       {:post [:underscore]}
   :normalise-attr         {:pre [:set
                                  :vector-attr]}
   :normalise-expression   {:post [(if (:expression pipeline) :expression)]}
   :normalise-single       {:pre  [:id]
                            :post [:list
                                   :symbol
                                   :vector-single]}})

(defn normalise
  "customised normalise pipeline for datomic given additional inputs
 
   (-> (datomic/connect! {:uri \"datomic:mem://datomic-analyse-test\"
                          :schema  {:user  {:email [{:required true}]}}})
       (assoc-in [:process :input] {:user/email \"a@a.com\"})
       (normalise)
       :process :normalised)
   => {:user {:email \"a@a.com\"}}"
  {:added "0.9"}
  [{:keys [pipeline] :as datasource}]
  (let [data (-> datasource :process :input)
        ndata (if (-> datasource :options :skip-normalise)
                data
                (->> (merge datomic-wrapper-fns pipeline/normalise-wrapper-fns)
                     (pipeline/normalise-wrappers datasource (datomic-additions datasource))
                     (pipeline/normalise data datasource)))]
    (assoc-in datasource [:process :normalised] ndata)))
