(ns spirit.io.datomic.schema.generate
  (:require [spirit.io.datomic.schema.base :as base]
            [hara.data.path :as data]
            [hara.string.path :as path]
            [hara.common.error :refer [error]]))

(defn datomic-attr-property
  "creates a datomic property description from spirit
   (datomic-attr-property {:type :string} :type
                          (base/datomic-specific :type) {})
   => {:db/valueType :db.type/string}
 
   (datomic-attr-property {:cardinality :one} :cardinality
                         (base/datomic-specific :cardinality) {})
   => {:db/cardinality :db.cardinality/one}
 
   (datomic-attr-property {} :cardinality
                         (base/datomic-specific :cardinality) {})
   => {:db/cardinality :db.cardinality/one}
 
   (datomic-attr-property {} :unique
                         (base/datomic-specific :unique) {})
   => {}"
  {:added "0.1"}
  [attr k pnuema res]
  (let [dft  (:default pnuema)
        v    (or (k attr) dft)
        prop-pair  (fn [attr k v f]
                     [(keyword (str "db/" (name attr)))
                      (f v k)])]
    (cond (nil? v)
          (if (:required pnuema)
            (error "DATOMIC-ATTR-PROPERTY: Property " k " is required")
            res)

          :else
          (let [chk  (or (:check pnuema) (constantly true))
                f    (or (:fn pnuema) (fn [x & xs] x))
                attr (or (:attr pnuema) k)]
            (if (not (chk v))
              (error  "DATOMIC-ATTR-PROPERTY: Property " v
                      " failed check on " attr " for check " chk)
              (apply assoc res (prop-pair attr k v f)))))))

(defn datomic-attr
  "creates a datomic attribute schema entry
   (datomic-attr [{:ident :account/name
                   :type  :string}])
   => (contains {:db.install/_attribute :db.part/db,
                 :db/index false,
                 :db/fulltext false,
                 :db/noHistory false,
                 :db/valueType :db.type/string,
                 :db/ident :account/name,
                 :db/cardinality :db.cardinality/one})
 
   (datomic-attr [{:ident       :account/tags
                   :type        :string
                   :cardinality :many
                   :fulltext    true
                   :index       true
                  :doc         \"tags for account\"}])
   => (contains {:db.install/_attribute :db.part/db
                 :db/ident        :account/tags
                 :db/index        true
                 :db/doc          \"tags for account\"
                 :db/valueType    :db.type/string
                 :db/fulltext     true
                 :db/cardinality  :db.cardinality/many})"
  {:added "0.1"}
  [[attr]]
  (reduce-kv (fn [out k v]
               (datomic-attr-property attr k v out))
             {:db.install/_attribute :db.part/db}
             base/datomic-specific))

(defn datomic-enum
  "creates datomic idents from an `:enum` attr
   (->> (datomic-enum [{:ident   :person/gender
                        :type    :enum
                        :enum    {:ns     :person.gender
                                  :values #{:male  :female}}}])
        (map #(dissoc % :db/id)))
   => [{:db/ident :person.gender/female}
       {:db/ident :person.gender/male}]"
  {:added "0.1"}
  [[attr]]
  (map (fn [v]
         {:db/ident (path/join [(-> attr :enum :ns) v])})
       (-> attr :enum :values)))

(defn datomic-schema
  "creates a full datomic schema from spirit
   (->> (datomic-schema {:node/link   [{:ident :node/link
                                        :type  :ref
                                        :ref {:ns  :node}}]
                         :person/gender [{:ident   :person/gender
                                          :type    :enum
                                          :enum    {:ns     :person.gender
                                                    :values #{:male  :female}}}]})
       (map #(dissoc % :db/id)))
   => [{:db.install/_attribute :db.part/db,
        :db/index false,
        :db/fulltext false,
        :db/noHistory false,
        :db/valueType
        :db.type/ref,
        :db/ident :node/link,
        :db/cardinality :db.cardinality/one}
       {:db.install/_attribute :db.part/db,
        :db/index false,
        :db/fulltext false,
        :db/noHistory false,
        :db/valueType :db.type/ref,
        :db/ident :person/gender,
        :db/cardinality :db.cardinality/one}
       {:db/ident :person.gender/female}
       {:db/ident :person.gender/male}]"
  {:added "0.1"}
  [essence]
  (let [attrs (-> essence
                  data/flatten-keys-nested)
        [enums attrs] (reduce-kv (fn [[enums rest] _ [{:keys [type]} :as attr]]
                                (if (= type :enum)
                                 [(conj enums attr) rest]
                                 [enums (conj rest attr)]))
                              [[] []]
                              attrs)
        attrs (mapv datomic-attr attrs)
        enum-attrs (->> enums
                        (map #(assoc-in % [0 :type] :ref))
                        (map datomic-attr))
        enum-data  (mapcat datomic-enum enums)]
    (concat attrs enum-attrs enum-data)))
