(ns spirit.mail.mock
  (:require [spirit.protocol.imail :as mail]
            [spirit.data.atom :as atom]
            [spirit.mail.common :as common]
            [spirit.mail.interop.message :as message]
            [hara.component :as component]
            [hara.object :as object]
            [clojure.set :as set])
  (:refer-clojure :exclude [send])
  (:import java.util.UUID))

(defonce ^:dynamic *active* (atom {}))

(defn activate
  "puts a given mailbox into the *active* record
 
   (-> (mail/create-mailbox {:type :mock
                             :id :a
                             :addresses #{\"a@a.com\"}})
       (activate))
 
   (:a @*active*)
   => spirit.mail.mock.MockMailbox"
  {:added "0.8"}
  ([mailbox]
   (let [id (or (:id mailbox)
                (str (UUID/randomUUID)))]
     (activate id mailbox)))
  ([id mailbox]
   (if-let [old (get @*active* id)]
     (component/-stop old))
   (swap! *active* assoc id mailbox)
   (assoc mailbox :id id)))

(defn deactivate
  "removes a given mailbox into the *active* record
 
   (-> (mail/create-mailbox {:type :mock
                             :id :a
                             :addresses #{\"a@a.com\"}})
       (activate)
       (deactivate))
 
   (:a @*active*) => nil"
  {:added "0.8"}
  [mailbox]
  (swap! *active* dissoc (:id mailbox))
  (dissoc mailbox :id))

(defrecord MockMailbox [id state addresses]
  Object
  (toString [_]
    (str "#mailbox.mock" (if (and state
                                  (get @*active* id))
                           {:id id
                            :messages (count @state)
                            :addresses  addresses}
                           "<uninitiased>")))
  
  mail/IMailbox
  (-init-mail [_ messages]
    (reset! state (mapv common/to-string messages)))
  
  (-count-mail [_]
    (count @state))
  
  (-get-mail [_ i]
    (common/from-string (nth @state i)))
  
  (-list-mail [_]
    (map common/from-string @state))

  (-clear-mail [_]
    (reset! state []))

  component/IComponent
  (-start [mailbox]
    (-> mailbox atom/attach-state activate))

  (-stop [mailbox]
    (-> mailbox deactivate atom/detach-state)))

(defmethod print-method MockMailbox
  [v w]
  (.write w (str v)))

(defmethod mail/create-mailbox :mock
  [m]
  (-> (assoc m :initial [])
      (update-in [:addresses] set)
      (map->MockMailbox)))

(defn send
  "sends a message to an active mailbox
 
   (-> (mail/create-mailbox {:type :mock
                             :id :a
                             :addresses #{\"a@a.com\"}})
       (component/start))
 
   (count @(:state (:a @*active*)))
   => 0
   
   (send {:to  [\"a@a.com\"]
          :subject \"Test\"
          :body \"Hello There\"})
   
   (count @(:state (:a @*active*)))
   => 1"
  {:added "0.8"}
  [message]
  (let [m   (message/message message)
        recipients (set (object/to-data (object/get m :all-recipients)))
        m   (doto m
              (.removeHeader "Bcc"))]
    
    (doseq [[_ mailbox] @*active*]
      (let [matching (count (set/intersection recipients (:addresses mailbox)))]
        (if (pos? matching)
          (swap! (:state mailbox) conj (common/to-string m)))))))

(defrecord MockMailer []
  Object
  (toString [_]
    (str "#mailer.mock" {}))

  mail/IMailer
  (mail/-send-mail [_ message]
    (send message))

  (mail/-send-bulk [_ messages]
    (doseq [m messages]
      (send m))))

(defmethod print-method MockMailer
  [v w]
  (.write w (str v)))

(defmethod mail/create-mailer :mock
  [m]
  (map->MockMailer m))
