(ns yin.test
  (:require [cljs.analyzer.api :as analyzer]
            [yin.test.checker.collection :as collection]
            [hara.namespace.import :as ns]))

(ns/import yin.test.checker.collection [just-in contains-in])

(defonce ^:dynamic *meta* nil)

(defonce ^:dynamic *desc* nil)

(defonce ^:dynamic *path* nil)

(declare =>) 

(def arrows '{=> :test-equal})

(defn split
  "creates a sequence of pairs from a loose sequence
   (split '[(def a 1)
            (+ a 3)
            => 5])
   => (contains-in [{:type :form,
                     :form '(def a 1)}
                    {:type :test-equal,
                     :input '(+ a 3),
                    :output 5}])"
  {:added "2.4"}
  ([body] (split body []))
  ([[x y z & more :as arr] out]
   (cond (empty? arr)
         out
         
         (get arrows y)
         (recur more
                (conj out {:type (get arrows y)
                           :meta (merge *meta*
                                        (or (meta x) (meta y) (meta z)))
                           :input {:form (list 'quote x)
                                   :function `(fn [] ~x)}
                           :output {:form (list 'quote z)
                                    :function `(fn [] ~z)}}))

         :else
         (recur (rest arr)
                (conj out {:type :form
                           :meta (merge *meta* (meta x))
                           :form (list 'quote x)
                           :function `(fn [] ~x)})))))

(defmacro fact
  [& [desc? & body]]
  (let [[desc body] (if (string? desc?)
                      [desc? body]
                      [nil (cons desc? body)])
        {:keys [as refer]} (meta &form)
        fmeta  {:ns (-> &env :ns :name str)
                :refer (list 'quote refer)
                :desc desc}
        sym    (or as
                   (if refer (symbol (str "test-" (name refer))))
                   (throw (Exception. (str "Requires a :refer or :as entry on the meta tag"))))]
    (binding [*meta* fmeta]
      `(do 
         (defn ~sym
           ~(merge {:fact true} (select-keys (meta &form) [:as :refer]))
           []
           (->> ~(split body)
                (mapv yin.test.form.process/process)
                (yin.test.form.process/collect ~*meta*)))
         (~sym)))))

(defn namespace-tests [ns defs]
  (->> defs
       (sort-by (comp :line :meta second))
       (filter  (comp :fact :meta second))
       (map first)
       (mapv (fn [name] (symbol (str ns) (str name))))))

(defn run-namespace-form
  ([ns]
   (let [test-ns (if (.endsWith (str ns) "-test")
                   ns
                   (symbol (str (str ns) "-test")))
         {:keys [defs name] :as env} (analyzer/find-ns test-ns)]
     `(do (println "\n")
          (println  (-> (str "---- Namespace (" ~(str name) ") ----")
                        (yin.io.ansii/style #{:blue :bold})))
          (yin.test.runner/run-tests
           ~(namespace-tests ns defs))))))

(defmacro run-namespace
  ([]
   (run-namespace-form (-> &env :ns :name)))
  ([ns]
   (run-namespace-form (eval ns))))

(defmacro run []
  (let [nss (sort (filter #(.endsWith (str %) "-test") (analyzer/all-ns)))
        inputs (mapv (juxt str
                           (fn [ns]
                             (->>(:defs (analyzer/find-ns ns))
                                 (namespace-tests ns)))) nss)]
    `(do (println (-> (str "\n---- Project (" (count ~inputs) " files) ----\n")
                      (yin.io.ansii/style #{:bold})))
         (yin.test.runner/run-namespace-tests ~inputs))))
