/*******************************************************************************
 * Copyright (c) 2011 Google, Inc.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * https://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Google, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.wb.internal.core.utils.check;

import org.eclipse.wb.internal.core.utils.exception.DesignerException;
import org.eclipse.wb.internal.core.utils.exception.ICoreExceptionConstants;

import java.text.MessageFormat;

/**
 * <code>Assert</code> is useful for for embedding runtime sanity checks in code. The predicate
 * methods all test a condition and throw some type of unchecked exception if the condition does not
 * hold.
 * <p>
 * Assertion failure exceptions, like most runtime exceptions, are thrown when something is
 * misbehaving. Assertion failures are invariably unspecified behavior; consequently, clients should
 * never rely on these being thrown (and certainly should not being catching them specifically).
 *
 * @author scheglov_ke
 * @coverage core.util
 */
public final class Assert {
	////////////////////////////////////////////////////////////////////////////
	//
	// Constructor
	//
	////////////////////////////////////////////////////////////////////////////
	private Assert() {
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// "legal"
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * Asserts that an argument is legal. If the given boolean is not <code>true</code>, an
	 * <code>IllegalArgumentException</code> is thrown.
	 *
	 * @param expression
	 *          the boolean expression of the check
	 * @return <code>true</code> if the check passes (does not return if the check fails)
	 * @exception IllegalArgumentException
	 *              if the legality test failed
	 */
	public static boolean isLegal(boolean expression) {
		return isLegal(expression, ""); //$NON-NLS-1$
	}

	/**
	 * Asserts that an argument is legal. If the given boolean is not <code>true</code>, an
	 * <code>IllegalArgumentException</code> is thrown. The given message is included in that
	 * exception, to aid debugging.
	 *
	 * @param expression
	 *          the boolean expression of the check
	 * @param message
	 *          the message to include in the exception
	 * @return <code>true</code> if the check passes (does not return if the check fails)
	 * @exception IllegalArgumentException
	 *              if the legality test failed
	 */
	public static boolean isLegal(boolean expression, String message) {
		if (!expression) {
			throw new IllegalArgumentException(message);
		}
		return expression;
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// "null"
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * Asserts that the given object is <code>null</code>. If this is not the case, some kind of
	 * unchecked exception is thrown.
	 *
	 * @param object
	 *          the value to test
	 */
	public static void isNull(Object object) {
		isNull(object, "null argument expected"); //$NON-NLS-1$
	}

	/**
	 * Asserts that the given object is <code>null</code>. If this is not the case, some kind of
	 * unchecked exception is thrown. The given message is included in that exception, to aid
	 * debugging.
	 *
	 * @param object
	 *          the value to test
	 * @param message
	 *          the message to include in the exception
	 */
	public static void isNull(Object object, String message) {
		if (object != null) {
			throw new AssertionFailedException(message);
		}
	}

	/**
	 * Asserts that the given object is not <code>null</code>. If this is not the case, some kind of
	 * unchecked exception is thrown. The given message is included in that exception, to aid
	 * debugging.
	 *
	 * @param object
	 *          the value to test
	 * @param errorFormat
	 *          the format of error message to produce if the check fails, as expected by
	 *          {@link String#format(String, Object...)}. For example
	 *          <code>"Execution flow problem. %s expected, but %s found."</code>.
	 * @param args
	 *          the arguments for {@code errorFormat}
	 */
	public static void isNull(Object object, String errorFormat, Object... args) {
		if (object != null) {
			String message = String.format(errorFormat, args);
			fail(message);
		}
	}

	/**
	 * @param errorFormat
	 *          the format of error message suitable for {@link MessageFormat}.
	 * @param errorFormat
	 *          the format of error message to produce if the check fails, as expected by
	 *          {@link MessageFormat}. For example
	 *          <code>"Execution flow problem. {0} expected, but {1} found."</code>.
	 */
	public static void isNull2(Object object, String errorFormat, Object... args) {
		if (object != null) {
			String message = MessageFormat.format(errorFormat, args);
			fail(message);
		}
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// not "null"
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * Asserts that the given object is not <code>null</code>. If this is not the case, some kind of
	 * unchecked exception is thrown.
	 *
	 * @param object
	 *          the value to test
	 */
	public static void isNotNull(Object object) {
		isNotNull(object, "null argument"); //$NON-NLS-1$
	}

	/**
	 * Asserts that the given object is not <code>null</code>. If this is not the case, some kind of
	 * unchecked exception is thrown. The given message is included in that exception, to aid
	 * debugging.
	 *
	 * @param object
	 *          the value to test
	 * @param message
	 *          the message to include in the exception
	 */
	public static void isNotNull(Object object, String message) {
		if (object == null) {
			fail(message);
		}
	}

	/**
	 * Asserts that the given object is not <code>null</code>. If this is not the case, some kind of
	 * unchecked exception is thrown. The given message is included in that exception, to aid
	 * debugging.
	 *
	 * @param object
	 *          the value to test
	 * @param errorFormat
	 *          the format of error message to produce if the check fails, as expected by
	 *          {@link String#format(String, Object...)}. For example
	 *          <code>"Execution flow problem. %s expected, but %s found."</code>.
	 * @param args
	 *          the arguments for {@code errorFormat}
	 */
	public static void isNotNull(Object object, String errorFormat, Object... args) {
		if (object == null) {
			fail(String.format(errorFormat, args));
		}
	}

	/**
	 * @param errorFormat
	 *          the format of error message suitable for {@link MessageFormat}.
	 * @param errorFormat
	 *          the format of error message to produce if the check fails, as expected by
	 *          {@link MessageFormat}. For example
	 *          <code>"Execution flow problem. {0} expected, but {1} found."</code>.
	 */
	public static void isNotNull2(Object object, String errorFormat, Object... args) {
		if (object == null) {
			String message = MessageFormat.format(errorFormat, args);
			fail(message);
		}
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// Fail
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * Fails with given message.
	 *
	 * @param message
	 *          the message to include in the exception
	 */
	public static void fail(String message) {
		throw new AssertionFailedException(message);
	}

	/**
	 * @param errorFormat
	 *          the format of error message to produce if the check fails, as expected by
	 *          {@link MessageFormat}. For example <code>"{0} expected, but {1} found."</code>.
	 */
	public static void fail(String errorFormat, Object... args) {
		String message = MessageFormat.format(errorFormat, args);
		throw new AssertionFailedException(message);
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// "true"
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * Asserts that the given boolean is <code>true</code>. If this is not the case, some kind of
	 * unchecked exception is thrown.
	 *
	 * @param expression
	 *          the boolean expression of the check
	 * @return <code>true</code> if the check passes (does not return if the check fails)
	 */
	public static boolean isTrue(boolean expression) {
		return isTrue(expression, "assertion failed"); //$NON-NLS-1$
	}

	/**
	 * Asserts that the given boolean is <code>true</code>. If this is not the case, some kind of
	 * unchecked exception is thrown. The given message is included in that exception, to aid
	 * debugging.
	 *
	 * @param expression
	 *          the boolean expression of the check
	 * @param message
	 *          the message to include in the exception
	 * @return <code>true</code> if the check passes (does not return if the check fails)
	 */
	public static boolean isTrue(boolean expression, String message) {
		if (!expression) {
			fail(message);
		}
		return expression;
	}

	/**
	 * Asserts that the given boolean is <code>true</code>. If this is not the case, some kind of
	 * unchecked exception is thrown. The given message is included in that exception, to aid
	 * debugging.
	 *
	 * @param expression
	 *          the boolean expression of the check
	 * @param errorFormat
	 *          the format of error message to produce if the check fails, as expected by
	 *          {@link String#format(String, Object...)}. For example
	 *          <code>"Execution flow problem. %s expected, but %s found."</code>.
	 * @param args
	 *          the arguments for {@code errorFormat}
	 * @return <code>true</code> if the check passes (does not return if the check fails)
	 */
	public static boolean isTrue(boolean expression, String errorFormat, Object... args) {
		if (!expression) {
			String message = String.format(errorFormat, args);
			fail(message);
		}
		return expression;
	}

	/**
	 * Asserts that the given boolean is <code>true</code>. If this is not the case, some kind of
	 * unchecked exception is thrown. The given message is included in that exception, to aid
	 * debugging.
	 *
	 * @param expression
	 *          the boolean expression to check.
	 * @param errorFormat
	 *          the format of error message to produce if the check fails, as expected by
	 *          {@link MessageFormat}. For example <code>"{0} expected, but {1} found."</code>.
	 */
	public static boolean isTrue2(boolean expression, String errorFormat, Object... args) {
		if (!expression) {
			fail(errorFormat, args);
		}
		return expression;
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// equals
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * Asserts that given actual value equals expected value. If this is not the case, some kind of
	 * unchecked exception is thrown.
	 *
	 * @param expected
	 *          the expected value
	 * @param the
	 *          actual value to check
	 */
	public static void equals(int expected, int actual) {
		equals(expected, actual, expected + " expected, but " + actual + " found");
	}

	/**
	 * Asserts that given actual value equals expected value. If this is not the case, some kind of
	 * unchecked exception is thrown. The given message is included in that exception, to aid
	 * debugging.
	 *
	 * @param expected
	 *          the expected value
	 * @param the
	 *          actual value to check
	 * @param message
	 *          the message to include in the exception
	 */
	public static void equals(int expected, int actual, String message) {
		if (expected != actual) {
			fail(message);
		}
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// instanceOf
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * Asserts that given object is not <code>null</code> and has class compatible with given.
	 */
	public static void instanceOf(Class<?> expectedClass, Object o) {
		if (o == null) {
			fail(expectedClass.getName() + " expected, but 'null' found.");
		}
		if (!expectedClass.isAssignableFrom(o.getClass())) {
			fail(expectedClass.getName() + " expected, but " + o.getClass().getName() + " found.");
		}
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// Designer exception checks
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * Asserts that the given boolean is <code>true</code>. If this is not the case,
	 * {@link DesignerException} is thrown.
	 *
	 * @param expression
	 *          the boolean expression of the check
	 * @param exceptionCode
	 *          the exception code, for example from {@link ICoreExceptionConstants}
	 * @param parameter
	 *          the parameter for {@link DesignerException}.
	 */
	public static void isTrueException(boolean expression, int exceptionCode, Object... parameters) {
		if (!expression) {
			String paramStrings[] = new String[parameters.length];
			for (int i = 0; i < parameters.length; i++) {
				Object parameter = parameters[i];
				paramStrings[i] = parameter != null ? parameter.toString() : "null";
			}
			throw new DesignerException(exceptionCode, paramStrings);
		}
	}
}
